"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createChatService = createChatService;
var _common = require("@kbn/kibana-utils-plugin/common");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _conversation_complete = require("../../common/conversation_complete");
var _types = require("../../common/types");
var _filter_function_definitions = require("../../common/utils/filter_function_definitions");
var _process_openai_stream = require("../../common/utils/process_openai_stream");
var _readable_stream_reader_into_observable = require("../utils/readable_stream_reader_into_observable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MIN_DELAY = 35;
function toObservable(response) {
  var _response$response, _response$response$bo;
  const status = (_response$response = response.response) === null || _response$response === void 0 ? void 0 : _response$response.status;
  if (!status || status >= 400) {
    var _response$response2;
    throw new Error(((_response$response2 = response.response) === null || _response$response2 === void 0 ? void 0 : _response$response2.statusText) || 'Unexpected error');
  }
  const reader = (_response$response$bo = response.response.body) === null || _response$response$bo === void 0 ? void 0 : _response$response$bo.getReader();
  if (!reader) {
    throw new Error('Could not get reader from response');
  }
  return (0, _readable_stream_reader_into_observable.readableStreamReaderIntoObservable)(reader).pipe(
  // append a timestamp of when each value was emitted
  (0, _rxjs.timestamp)(),
  // use the previous timestamp to calculate a target
  // timestamp for emitting the next value
  (0, _rxjs.scan)((acc, value) => {
    const lastTimestamp = acc.timestamp || 0;
    const emitAt = Math.max(lastTimestamp + MIN_DELAY, value.timestamp);
    return {
      timestamp: emitAt,
      value: value.value
    };
  }),
  // add the delay based on the elapsed time
  // using concatMap(of(value).pipe(delay(50))
  // leads to browser issues because timers
  // are throttled when the tab is not active
  (0, _rxjs.concatMap)(value => {
    const now = Date.now();
    const delayFor = value.timestamp - now;
    if (delayFor <= 0) {
      return (0, _rxjs.of)(value.value);
    }
    return (0, _rxjs.of)(value.value).pipe((0, _rxjs.delay)(delayFor));
  }));
}
async function createChatService({
  analytics,
  signal: setupAbortSignal,
  registrations,
  client
}) {
  const functionRegistry = new Map();
  const renderFunctionRegistry = new Map();
  const [{
    functionDefinitions,
    contextDefinitions
  }] = await Promise.all([client('GET /internal/observability_ai_assistant/functions', {
    signal: setupAbortSignal
  }), ...registrations.map(registration => {
    return registration({
      registerRenderFunction: (name, renderFn) => {
        renderFunctionRegistry.set(name, renderFn);
      }
    });
  })]);
  functionDefinitions.forEach(fn => {
    functionRegistry.set(fn.name, fn);
  });
  const getFunctions = options => {
    return (0, _filter_function_definitions.filterFunctionDefinitions)({
      ...options,
      definitions: functionDefinitions
    });
  };
  return {
    analytics,
    renderFunction: (name, args, response) => {
      var _response$content, _response$data;
      const fn = renderFunctionRegistry.get(name);
      if (!fn) {
        throw new Error(`Function ${name} not found`);
      }
      const parsedArguments = args ? JSON.parse(args) : {};
      const parsedResponse = {
        content: JSON.parse((_response$content = response.content) !== null && _response$content !== void 0 ? _response$content : '{}'),
        data: JSON.parse((_response$data = response.data) !== null && _response$data !== void 0 ? _response$data : '{}')
      };
      return fn === null || fn === void 0 ? void 0 : fn({
        response: parsedResponse,
        arguments: parsedArguments
      });
    },
    getContexts: () => contextDefinitions,
    getFunctions,
    hasFunction: name => {
      return functionRegistry.has(name);
    },
    hasRenderFunction: name => {
      return renderFunctionRegistry.has(name);
    },
    complete({
      connectorId,
      messages,
      conversationId,
      persist,
      signal
    }) {
      const subject = new _rxjs.Subject();
      client('POST /internal/observability_ai_assistant/chat/complete', {
        params: {
          body: {
            messages,
            connectorId,
            conversationId,
            persist
          }
        },
        signal,
        asResponse: true,
        rawResponse: true
      }).then(_response => {
        const response = _response;
        const response$ = toObservable(response).pipe((0, _rxjs.map)(line => JSON.parse(line)), (0, _rxjs.tap)(event => {
          if (event.type === _conversation_complete.StreamingChatResponseEventType.ConversationCompletionError) {
            var _event$error$code;
            const code = (_event$error$code = event.error.code) !== null && _event$error$code !== void 0 ? _event$error$code : _conversation_complete.ChatCompletionErrorCode.InternalError;
            const message = event.error.message;
            throw new _conversation_complete.ConversationCompletionError(code, message);
          }
        })).subscribe(subject);
        signal.addEventListener('abort', () => {
          response$.unsubscribe();
        });
      }).catch(err => {
        subject.error(err);
        subject.complete();
      });
      return subject;
    },
    chat({
      connectorId,
      messages,
      function: callFunctions = 'auto'
    }) {
      const subject = new _rxjs.BehaviorSubject({
        message: {
          role: _types.MessageRole.Assistant
        }
      });
      const contexts = ['core', 'apm'];
      const functions = getFunctions({
        contexts
      });
      const controller = new AbortController();
      client('POST /internal/observability_ai_assistant/chat', {
        params: {
          body: {
            messages,
            connectorId,
            functions: callFunctions === 'none' ? [] : functions.filter(fn => fn.visibility !== _types.FunctionVisibility.User).map(fn => (0, _lodash.pick)(fn, 'name', 'description', 'parameters'))
          }
        },
        signal: controller.signal,
        asResponse: true,
        rawResponse: true
      }).then(_response => {
        const response = _response;
        const subscription = toObservable(response).pipe((0, _process_openai_stream.processOpenAiStream)(),
        // merge the messages
        (0, _rxjs.scan)((acc, {
          choices
        }) => {
          var _choices$0$delta$cont, _choices$0$delta$func, _choices$0$delta$func2, _choices$0$delta$func3, _choices$0$delta$func4;
          acc.message.content += (_choices$0$delta$cont = choices[0].delta.content) !== null && _choices$0$delta$cont !== void 0 ? _choices$0$delta$cont : '';
          acc.message.function_call.name += (_choices$0$delta$func = (_choices$0$delta$func2 = choices[0].delta.function_call) === null || _choices$0$delta$func2 === void 0 ? void 0 : _choices$0$delta$func2.name) !== null && _choices$0$delta$func !== void 0 ? _choices$0$delta$func : '';
          acc.message.function_call.arguments += (_choices$0$delta$func3 = (_choices$0$delta$func4 = choices[0].delta.function_call) === null || _choices$0$delta$func4 === void 0 ? void 0 : _choices$0$delta$func4.arguments) !== null && _choices$0$delta$func3 !== void 0 ? _choices$0$delta$func3 : '';
          return (0, _lodash.cloneDeep)(acc);
        }, {
          message: {
            content: '',
            function_call: {
              name: '',
              arguments: '',
              trigger: _types.MessageRole.Assistant
            },
            role: _types.MessageRole.Assistant
          }
        }),
        // convert an error into state
        (0, _rxjs.catchError)(error => (0, _rxjs.of)({
          ...subject.value,
          error,
          aborted: error instanceof _common.AbortError || controller.signal.aborted
        }))).subscribe(subject);

        // if the request is aborted, convert that into state as well
        controller.signal.addEventListener('abort', () => {
          subscription.unsubscribe();
          subject.next({
            ...subject.value,
            aborted: true
          });
          subject.complete();
        });
      }).catch(async err => {
        if ('response' in err) {
          var _err$response;
          const body = await ((_err$response = err.response) === null || _err$response === void 0 ? void 0 : _err$response.json());
          err.body = body;
          if (body.message) {
            err.message = body.message;
          }
        }
        throw err;
      }).catch(err => {
        subject.next({
          ...subject.value,
          aborted: false,
          error: err
        });
        subject.complete();
      });
      const pendingMessages$ = subject.pipe(
      // make sure the request is only triggered once,
      // even with multiple subscribers
      (0, _rxjs.shareReplay)(1),
      // if the Observable is no longer subscribed,
      // abort the running request
      (0, _rxjs.finalize)(() => {
        controller.abort();
      }));
      return pendingMessages$;
    }
  };
}