"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metricByField = void 0;
exports.usePodMetricsTable = usePodMetricsTable;
var _react = require("react");
var _shared = require("../shared");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const podMetricsQueryConfig = {
  sourceFilter: {
    term: {
      'event.dataset': 'kubernetes.pod'
    }
  },
  groupByField: ['kubernetes.pod.uid', 'kubernetes.pod.name'],
  metricsMap: {
    'kubernetes.pod.cpu.usage.limit.pct': {
      aggregation: 'avg',
      field: 'kubernetes.pod.cpu.usage.limit.pct'
    },
    'kubernetes.pod.memory.usage.bytes': {
      aggregation: 'avg',
      field: 'kubernetes.pod.memory.usage.bytes'
    }
  }
};
const metricByField = (0, _shared.createMetricByFieldLookup)(podMetricsQueryConfig.metricsMap);
exports.metricByField = metricByField;
const unpackMetric = (0, _shared.makeUnpackMetric)(metricByField);
function usePodMetricsTable({
  timerange,
  filterClauseDsl,
  metricsClient
}) {
  const [currentPageIndex, setCurrentPageIndex] = (0, _react.useState)(0);
  const [sortState, setSortState] = (0, _react.useState)({
    field: 'averageCpuUsagePercent',
    direction: 'desc'
  });
  const {
    options: podMetricsOptions
  } = (0, _react.useMemo)(() => (0, _shared.metricsToApiOptions)(podMetricsQueryConfig, filterClauseDsl), [filterClauseDsl]);
  const {
    data,
    isLoading
  } = (0, _shared.useInfrastructureNodeMetrics)({
    metricsExplorerOptions: podMetricsOptions,
    timerange,
    transform: seriesToPodNodeMetricsRow,
    sortState,
    currentPageIndex,
    metricsClient
  });
  return {
    currentPageIndex,
    data,
    isLoading,
    setCurrentPageIndex,
    setSortState,
    sortState,
    timerange
  };
}
function seriesToPodNodeMetricsRow(series) {
  var _series$keys;
  const [id, name] = (_series$keys = series.keys) !== null && _series$keys !== void 0 ? _series$keys : [];
  if (series.rows.length === 0) {
    return rowWithoutMetrics(id, name);
  }
  return {
    id,
    name,
    ...calculateMetricAverages(series.rows)
  };
}
function rowWithoutMetrics(id, name) {
  return {
    id,
    name,
    averageCpuUsagePercent: null,
    averageMemoryUsageMegabytes: null
  };
}
function calculateMetricAverages(rows) {
  const {
    averageCpuUsagePercentValues,
    averageMemoryUsageMegabytesValues
  } = collectMetricValues(rows);
  let averageCpuUsagePercent = null;
  if (averageCpuUsagePercentValues.length !== 0) {
    averageCpuUsagePercent = (0, _shared.scaleUpPercentage)((0, _shared.averageOfValues)(averageCpuUsagePercentValues));
  }
  let averageMemoryUsageMegabytes = null;
  if (averageMemoryUsageMegabytesValues.length !== 0) {
    const averageInBytes = (0, _shared.averageOfValues)(averageMemoryUsageMegabytesValues);
    const bytesPerMegabyte = 1000000;
    averageMemoryUsageMegabytes = Math.floor(averageInBytes / bytesPerMegabyte);
  }
  return {
    averageCpuUsagePercent,
    averageMemoryUsageMegabytes
  };
}
function collectMetricValues(rows) {
  const averageCpuUsagePercentValues = [];
  const averageMemoryUsageMegabytesValues = [];
  rows.forEach(row => {
    const {
      averageCpuUsagePercent,
      averageMemoryUsageMegabytes
    } = unpackMetrics(row);
    if (averageCpuUsagePercent !== null) {
      averageCpuUsagePercentValues.push(averageCpuUsagePercent);
    }
    if (averageMemoryUsageMegabytes !== null) {
      averageMemoryUsageMegabytesValues.push(averageMemoryUsageMegabytes);
    }
  });
  return {
    averageCpuUsagePercentValues,
    averageMemoryUsageMegabytesValues
  };
}
function unpackMetrics(row) {
  return {
    averageCpuUsagePercent: unpackMetric(row, 'kubernetes.pod.cpu.usage.limit.pct'),
    averageMemoryUsageMegabytes: unpackMetric(row, 'kubernetes.pod.memory.usage.bytes')
  };
}