"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transparentize = exports.tintOrShade = exports.switchProp = exports.ifProp = exports.getContrast = exports.clampValue = exports.chooseLightOrDarkColor = void 0;
var _lodash = require("lodash");
var _polished = require("polished");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const asPropReader = reader => typeof reader === 'function' ? reader : (props, defaultValue) => (0, _lodash.get)(props, reader, defaultValue);
const switchProp = Object.assign((propName, options) => props => {
  const propValue = asPropReader(propName)(props, switchProp.default);
  if (typeof propValue === 'undefined') {
    return;
  }
  return options instanceof Map ? options.get(propValue) : (0, _lodash.get)(options, propValue);
}, {
  default: Symbol('default')
});
exports.switchProp = switchProp;
const ifProp = (propName, pass, fail) => props => asPropReader(propName)(props) ? pass : fail;
exports.ifProp = ifProp;
const tintOrShade = (textColor, color, tintFraction, shadeFraction) => {
  if ((0, _polished.parseToHsl)(textColor).lightness > 0.5) {
    return (0, _polished.shade)(1 - shadeFraction, color);
  } else {
    return (0, _polished.tint)(1 - tintFraction, color);
  }
};
exports.tintOrShade = tintOrShade;
const getContrast = (color1, color2) => {
  const luminance1 = (0, _polished.getLuminance)(color1);
  const luminance2 = (0, _polished.getLuminance)(color2);
  return parseFloat((luminance1 > luminance2 ? (luminance1 + 0.05) / (luminance2 + 0.05) : (luminance2 + 0.05) / (luminance1 + 0.05)).toFixed(2));
};
exports.getContrast = getContrast;
const chooseLightOrDarkColor = (backgroundColor, lightColor, darkColor) => {
  if (getContrast(backgroundColor, lightColor) > getContrast(backgroundColor, darkColor)) {
    return lightColor;
  } else {
    return darkColor;
  }
};
exports.chooseLightOrDarkColor = chooseLightOrDarkColor;
const transparentize = (amount, color) => {
  if (color === 'transparent') {
    return color;
  }
  const parsedColor = (0, _polished.parseToRgb)(color);
  const alpha = 'alpha' in parsedColor && typeof parsedColor.alpha === 'number' ? parsedColor.alpha : 1;
  const colorWithAlpha = {
    ...parsedColor,
    alpha: clampValue((alpha * 100 - amount * 100) / 100, 0, 1)
  };
  return (0, _polished.rgba)(colorWithAlpha);
};
exports.transparentize = transparentize;
const clampValue = (value, minValue, maxValue) => Math.max(minValue, Math.min(maxValue, value));
exports.clampValue = clampValue;