"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformOsType = exports.removeExpiredExceptions = exports.getBaseNestedClause = exports.getBaseMatchAnyClause = exports.filterOutUnprocessableValueLists = exports.createOrClauses = exports.createInnerAndClauses = exports.chunkExceptions = exports.buildNestedClause = exports.buildMatchWildcardClause = exports.buildMatchClause = exports.buildMatchAnyClause = exports.buildListClause = exports.buildIpRangeClauses = exports.buildExistsClause = exports.buildExclusionClause = exports.buildExceptionItemFilterWithOsType = exports.buildExceptionItemFilter = exports.buildExceptionFilter = void 0;
var _fp = require("lodash/fp");
var _securitysolutionIoTsListTypes = require("@kbn/securitysolution-io-ts-list-types");
var _lodash = require("lodash");
var _securitysolutionListUtils = require("@kbn/securitysolution-list-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const chunkExceptions = (exceptions, chunkSize) => {
  return (0, _fp.chunk)(chunkSize, exceptions);
};

/**
 * Transforms the os_type into a regular filter as if the user had created it
 * from the fields for the next state of transforms which will create the elastic filters
 * from it.
 *
 * Note: We use two types of fields, the "host.os.type" and "host.os.name.caseless"
 * The endpoint/endgame agent has been using "host.os.name.caseless" as the same value as the ECS
 * value of "host.os.type" where the auditbeat, winlogbeat, etc... (other agents) are all using
 * "host.os.type". In order to be compatible with both, I create an "OR" between these two data types
 * where if either has a match then we will exclude it as part of the match. This should also be
 * forwards compatible for endpoints/endgame agents when/if they upgrade to using "host.os.type"
 * rather than using "host.os.name.caseless" values.
 *
 * Also we create another "OR" from the osType names so that if there are multiples such as ['windows', 'linux']
 * this will exclude anything with either 'windows' or with 'linux'
 * @param osTypes The os_type array from the REST interface that is an array such as ['windows', 'linux']
 * @param entries The entries to join the OR's with before the elastic filter change out
 */
exports.chunkExceptions = chunkExceptions;
const transformOsType = (osTypes, entries) => {
  const hostTypeTransformed = osTypes.map(osType => {
    return [{
      field: 'host.os.type',
      operator: 'included',
      type: 'match',
      value: osType
    }, ...entries];
  });
  const caseLessTransformed = osTypes.map(osType => {
    return [{
      field: 'host.os.name.caseless',
      operator: 'included',
      type: 'match',
      value: osType
    }, ...entries];
  });
  return [...hostTypeTransformed, ...caseLessTransformed];
};

/**
 * This builds an exception item filter with the os type
 * @param osTypes The os_type array from the REST interface that is an array such as ['windows', 'linux']
 * @param entries The entries to join the OR's with before the elastic filter change out
 */
exports.transformOsType = transformOsType;
const buildExceptionItemFilterWithOsType = async (osTypes, entries, listClient) => {
  let isUnprocessable = false;
  const entriesWithOsTypes = transformOsType(osTypes, entries);
  const exceptionItemFilter = [];
  await Promise.all(entriesWithOsTypes.map(async entryWithOsType => {
    const esFilter = [];
    await Promise.all(entryWithOsType.map(async entry => {
      const filter = await createInnerAndClauses({
        entry,
        listClient
      });
      if (!filter) {
        isUnprocessable = true;
        return;
      }
      esFilter.push(filter);
    }));
    exceptionItemFilter.push({
      bool: {
        filter: esFilter
      }
    });
  }));
  return isUnprocessable ? undefined : exceptionItemFilter;
};
exports.buildExceptionItemFilterWithOsType = buildExceptionItemFilterWithOsType;
const buildExceptionItemFilter = async (exceptionItem, listClient) => {
  const {
    entries,
    os_types: osTypes
  } = exceptionItem;
  if (osTypes != null && osTypes.length > 0) {
    return buildExceptionItemFilterWithOsType(osTypes, entries, listClient);
  } else {
    if (entries.length === 1) {
      const filter = await createInnerAndClauses({
        entry: entries[0],
        listClient
      });
      if (!filter) {
        return undefined;
      }
      return [filter];
    } else {
      const esFilter = [];
      for (const entry of entries) {
        const filter = await createInnerAndClauses({
          entry,
          listClient
        });
        if (!filter) {
          return undefined;
        }
        esFilter.push(filter);
      }
      return [{
        bool: {
          filter: esFilter
        }
      }];
    }
  }
};
exports.buildExceptionItemFilter = buildExceptionItemFilter;
const createOrClauses = async ({
  exceptionsWithoutValueLists,
  exceptionsWithValueLists,
  chunkSize,
  listClient
}) => {
  const unprocessableExceptionItems = [];
  const orClauses = [];
  for (const exceptionItem of exceptionsWithoutValueLists) {
    const filter = await buildExceptionItemFilter(exceptionItem, listClient);
    if (!filter) {
      unprocessableExceptionItems.push(exceptionItem);
    } else {
      orClauses.push(filter);
    }
  }

  // Chunk the exceptions that will require list client requests
  const chunks = chunkExceptions(exceptionsWithValueLists, chunkSize);
  for (const exceptionsChunk of chunks) {
    await Promise.all(exceptionsChunk.map(async exceptionItem => {
      const filter = await buildExceptionItemFilter(exceptionItem, listClient);
      if (!filter) {
        unprocessableExceptionItems.push(exceptionItem);
        return;
      }
      orClauses.push(filter);
    }));
  }
  return {
    orClauses: orClauses.flat(),
    unprocessableExceptionItems
  };
};
exports.createOrClauses = createOrClauses;
const isListTypeProcessable = type => type === 'keyword' || type === 'ip' || type === 'ip_range';
const filterOutUnprocessableValueLists = async (exceptionItems, listClient) => {
  const exceptionBooleans = await Promise.all(exceptionItems.map(async exceptionItem => {
    const listEntries = exceptionItem.entries.filter(entry => _securitysolutionIoTsListTypes.entriesList.is(entry));
    for await (const listEntry of listEntries) {
      const {
        list: {
          id,
          type
        }
      } = listEntry;
      if (!isListTypeProcessable(type)) {
        return false;
      }

      // Don't want any items, just the total list size
      const valueList = await listClient.findListItem({
        currentIndexPosition: 0,
        filter: '',
        listId: id,
        page: 0,
        perPage: 0,
        runtimeMappings: undefined,
        searchAfter: [],
        sortField: undefined,
        sortOrder: undefined
      });
      if (!valueList || valueList && valueList.total > _securitysolutionListConstants.MAXIMUM_SMALL_VALUE_LIST_SIZE) {
        return false;
      }
    }
    // If we're here, all the entries are processable
    return true;
  }));
  const filteredExceptions = exceptionItems.filter((item, index) => exceptionBooleans[index]);
  const unprocessableValueListExceptions = exceptionItems.filter((item, index) => !exceptionBooleans[index]);
  return {
    filteredExceptions,
    unprocessableValueListExceptions
  };
};
exports.filterOutUnprocessableValueLists = filterOutUnprocessableValueLists;
const removeExpiredExceptions = (lists, startedAt) => lists.filter(listItem => {
  if (listItem.expire_time && new Date(listItem.expire_time) < startedAt) {
    return false;
  }
  return true;
});
exports.removeExpiredExceptions = removeExpiredExceptions;
const buildExceptionFilter = async ({
  lists,
  excludeExceptions,
  chunkSize,
  alias = null,
  listClient,
  startedAt
}) => {
  const filteredLists = removeExpiredExceptions(lists, startedAt);

  // Remove exception items with large value lists. These are evaluated
  // elsewhere for the moment being.
  const [exceptionsWithoutValueLists, valueListExceptions] = (0, _lodash.partition)(filteredLists, item => !(0, _securitysolutionListUtils.hasLargeValueList)(item.entries));

  // Exceptions that contain large value list exceptions and will be processed later on in rule execution
  const unprocessedExceptions = [];
  const {
    filteredExceptions: exceptionsWithValueLists,
    unprocessableValueListExceptions
  } = await filterOutUnprocessableValueLists(valueListExceptions, listClient);
  unprocessedExceptions.push(...unprocessableValueListExceptions);
  if (exceptionsWithoutValueLists.length === 0 && exceptionsWithValueLists.length === 0) {
    return {
      filter: undefined,
      unprocessedExceptions
    };
  }
  const {
    orClauses,
    unprocessableExceptionItems
  } = await createOrClauses({
    chunkSize,
    exceptionsWithValueLists,
    exceptionsWithoutValueLists,
    listClient
  });
  const exceptionFilter = {
    meta: {
      alias,
      disabled: false,
      negate: excludeExceptions
    },
    query: {
      bool: {
        should: orClauses
      }
    }
  };
  unprocessedExceptions.concat(unprocessableExceptionItems);
  return {
    filter: exceptionFilter,
    unprocessedExceptions
  };
};
exports.buildExceptionFilter = buildExceptionFilter;
const buildExclusionClause = booleanFilter => {
  return {
    bool: {
      must_not: booleanFilter
    }
  };
};
exports.buildExclusionClause = buildExclusionClause;
const buildMatchClause = entry => {
  const {
    field,
    operator,
    value
  } = entry;
  const matchClause = {
    bool: {
      minimum_should_match: 1,
      should: [{
        match_phrase: {
          [field]: value
        }
      }]
    }
  };
  if (operator === 'excluded') {
    return buildExclusionClause(matchClause);
  } else {
    return matchClause;
  }
};
exports.buildMatchClause = buildMatchClause;
const getBaseMatchAnyClause = entry => {
  const {
    field,
    value
  } = entry;
  if (value.length === 1) {
    return {
      bool: {
        minimum_should_match: 1,
        should: [{
          match_phrase: {
            [field]: value[0]
          }
        }]
      }
    };
  }
  return {
    bool: {
      minimum_should_match: 1,
      should: value.map(val => {
        return {
          bool: {
            minimum_should_match: 1,
            should: [{
              match_phrase: {
                [field]: val
              }
            }]
          }
        };
      })
    }
  };
};
exports.getBaseMatchAnyClause = getBaseMatchAnyClause;
const buildMatchAnyClause = entry => {
  const {
    operator
  } = entry;
  const matchAnyClause = getBaseMatchAnyClause(entry);
  if (operator === 'excluded') {
    return buildExclusionClause(matchAnyClause);
  } else {
    return matchAnyClause;
  }
};
exports.buildMatchAnyClause = buildMatchAnyClause;
const buildMatchWildcardClause = entry => {
  const {
    field,
    operator,
    value
  } = entry;
  const wildcardClause = {
    bool: {
      filter: {
        wildcard: {
          [field]: value
        }
      }
    }
  };
  if (operator === 'excluded') {
    return buildExclusionClause(wildcardClause);
  } else {
    return wildcardClause;
  }
};
exports.buildMatchWildcardClause = buildMatchWildcardClause;
const buildExistsClause = entry => {
  const {
    field,
    operator
  } = entry;
  const existsClause = {
    bool: {
      minimum_should_match: 1,
      should: [{
        exists: {
          field
        }
      }]
    }
  };
  if (operator === 'excluded') {
    return buildExclusionClause(existsClause);
  } else {
    return existsClause;
  }
};
exports.buildExistsClause = buildExistsClause;
const isBooleanFilter = clause => {
  if (!clause) {
    return false;
  }
  const keys = Object.keys(clause);
  return keys.includes('bool') != null;
};
const buildIpRangeClauses = (ranges, field) => ranges.map(range => {
  const [gte, lte] = range.split('-');
  return {
    range: {
      [field]: {
        gte,
        lte
      }
    }
  };
});
exports.buildIpRangeClauses = buildIpRangeClauses;
const buildListClause = async (entry, listClient) => {
  const {
    field,
    operator,
    list: {
      type
    }
  } = entry;
  const list = await listClient.findAllListItems({
    filter: '',
    listId: entry.list.id
  });
  if (list == null) {
    throw new TypeError(`Cannot find list: "${entry.list.id}"`);
  }
  const listValues = list.data.map(listItem => listItem.value);
  if (type === 'ip_range') {
    const [dashNotationRange, slashNotationRange] = (0, _lodash.partition)(listValues, value => {
      return value.includes('-');
    });
    if (dashNotationRange.length > _securitysolutionListConstants.MAXIMUM_SMALL_IP_RANGE_VALUE_LIST_DASH_SIZE) {
      return undefined;
    }
    const rangeClauses = buildIpRangeClauses(dashNotationRange, field);
    if (slashNotationRange.length > 0) {
      rangeClauses.push({
        terms: {
          [field]: slashNotationRange
        }
      });
    }
    return {
      bool: {
        [operator === 'excluded' ? 'must_not' : 'should']: rangeClauses,
        minimum_should_match: 1
      }
    };
  }
  return {
    bool: {
      [operator === 'excluded' ? 'must_not' : 'filter']: {
        terms: {
          [field]: listValues
        }
      }
    }
  };
};
exports.buildListClause = buildListClause;
const getBaseNestedClause = async (entries, parentField, listClient) => {
  if (entries.length === 1) {
    const [singleNestedEntry] = entries;
    const innerClause = await createInnerAndClauses({
      entry: singleNestedEntry,
      listClient,
      parent: parentField
    });
    return isBooleanFilter(innerClause) ? innerClause : {
      bool: {}
    };
  }
  const filter = [];
  let isUnprocessable = false;
  await Promise.all(entries.map(async nestedEntry => {
    const clauses = await createInnerAndClauses({
      entry: nestedEntry,
      listClient,
      parent: parentField
    });
    if (!clauses) {
      isUnprocessable = true;
      return;
    }
    filter.push(clauses);
  }));
  if (isUnprocessable) {
    return undefined;
  }
  return {
    bool: {
      filter
    }
  };
};
exports.getBaseNestedClause = getBaseNestedClause;
const buildNestedClause = async (entry, listClient) => {
  const {
    field,
    entries
  } = entry;
  const baseNestedClause = await getBaseNestedClause(entries, field, listClient);
  if (!baseNestedClause) {
    return undefined;
  }
  return {
    nested: {
      path: field,
      query: baseNestedClause,
      score_mode: 'none'
    }
  };
};
exports.buildNestedClause = buildNestedClause;
const createInnerAndClauses = async ({
  entry,
  parent,
  listClient
}) => {
  const field = parent != null ? `${parent}.${entry.field}` : entry.field;
  if (_securitysolutionIoTsListTypes.entriesExists.is(entry)) {
    return buildExistsClause({
      ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesMatch.is(entry)) {
    return buildMatchClause({
      ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesMatchAny.is(entry)) {
    return buildMatchAnyClause({
      ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesMatchWildcard.is(entry)) {
    return buildMatchWildcardClause({
      ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesList.is(entry)) {
    return buildListClause({
      ...entry,
      field
    }, listClient);
  } else if (_securitysolutionIoTsListTypes.entriesNested.is(entry)) {
    return buildNestedClause(entry, listClient);
  } else {
    throw new TypeError(`Unexpected exception entry: ${entry}`);
  }
};
exports.createInnerAndClauses = createInnerAndClauses;