"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkForDuplicateTitle = checkForDuplicateTitle;
var _constants = require("../../../common/constants");
var _constants2 = require("./constants");
var _find_object_by_title = require("./find_object_by_title");
var _display_duplicate_title_confirm_modal = require("./display_duplicate_title_confirm_modal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * check for an existing saved object with the same title in ES
 * returns Promise<true> when it's no duplicate, or the modal displaying the warning
 * that's there's a duplicate is confirmed, else it returns a rejected Promise<ErrorMsg>
 */
async function checkForDuplicateTitle(savedObjectMeta, onTitleDuplicate, services) {
  const {
    client,
    overlays
  } = services;
  const {
    id,
    title,
    isTitleDuplicateConfirmed,
    lastSavedTitle,
    copyOnSave
  } = savedObjectMeta;

  // Don't check for duplicates if user has already confirmed save with duplicate title
  if (isTitleDuplicateConfirmed) {
    return true;
  }

  // Don't check if the user isn't updating the title, otherwise that would become very annoying to have
  // to confirm the save every time, except when copyOnSave is true, then we do want to check.
  if (title === lastSavedTitle && !copyOnSave) {
    return true;
  }
  const duplicate = await (0, _find_object_by_title.findObjectByTitle)(client, _constants.DOC_TYPE, title);
  if (!duplicate || duplicate.id === id) {
    return true;
  }
  if (onTitleDuplicate) {
    onTitleDuplicate();
    return Promise.reject(new Error(_constants2.SAVE_DUPLICATE_REJECTED));
  }

  // TODO: make onTitleDuplicate a required prop and remove UI components from this class
  // Need to leave here until all users pass onTitleDuplicate.
  return (0, _display_duplicate_title_confirm_modal.displayDuplicateTitleConfirmModal)(savedObjectMeta, overlays);
}