"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unsafeProcessorMove = exports.isOnFailureSelector = exports.isChildPath = exports.duplicateProcessor = exports.PARENT_CHILD_NEST_ERROR = void 0;
var _uuid = require("uuid");
var _constants = require("../constants");
var _utils = require("../utils");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * We know that it must be an on-failure handler if the selector length is greater than 2
 * because the first element will always be either processors or the global on-failure
 * array and the second element will be a number indicating the processor position in the
 * array. Anything more than that we know we are add an on failure handler.
 */
const isOnFailureSelector = selector => selector[0] === _constants2.ON_FAILURE_STATE_SCOPE || selector.length > 2;
exports.isOnFailureSelector = isOnFailureSelector;
const PARENT_CHILD_NEST_ERROR = 'PARENT_CHILD_NEST_ERROR';
exports.PARENT_CHILD_NEST_ERROR = PARENT_CHILD_NEST_ERROR;
const duplicateProcessor = sourceProcessor => {
  const onFailure = sourceProcessor.onFailure ? sourceProcessor.onFailure.map(p => duplicateProcessor(p)) : undefined;
  return {
    ...sourceProcessor,
    onFailure,
    id: (0, _uuid.v4)(),
    options: {
      ...sourceProcessor.options
    }
  };
};
exports.duplicateProcessor = duplicateProcessor;
const isChildPath = (a, b) => {
  return a.every((pathSegment, idx) => pathSegment === b[idx]);
};

/**
 * Unsafe!
 *
 * This function takes a data structure and mutates it in place.
 *
 * It is convenient for updating the processors (see {@link ProcessorInternal})
 * structure in this way because the structure is recursive. We are moving processors between
 * different arrays, removing in one, and adding to another. The end result should be consistent
 * with these actions.
 *
 * @remark
 * This function assumes parents cannot be moved into themselves.
 */
exports.isChildPath = isChildPath;
const unsafeProcessorMove = (state, source, destination) => {
  const pathToSourceArray = source.slice(0, -1);
  const pathToDestArray = destination.slice(0, -1);
  if (isChildPath(source, destination)) {
    throw new Error(PARENT_CHILD_NEST_ERROR);
  }
  const isXArrayMove = !(0, _utils.checkIfSamePath)(pathToSourceArray, pathToDestArray);

  // Start by setting up references to objects of interest using our selectors
  // At this point, our selectors are consistent with the data passed in.
  const sourceProcessors = (0, _utils.getValue)(pathToSourceArray, state);
  const destinationProcessors = (0, _utils.getValue)(pathToDestArray, state);
  const sourceIndex = parseInt(source[source.length - 1], 10);
  const sourceProcessor = (0, _utils.getValue)(pathToSourceArray.slice(0, -1), state);
  const processor = sourceProcessors[sourceIndex];
  const lastDestItem = destination[destination.length - 1];
  let destIndex;
  if (lastDestItem === _constants.DropSpecialLocations.top) {
    destIndex = 0;
  } else if (lastDestItem === _constants.DropSpecialLocations.bottom) {
    destIndex = Infinity;
  } else if (/^-?[0-9]+$/.test(lastDestItem)) {
    destIndex = parseInt(lastDestItem, 10);
  } else {
    throw new Error(`Expected number but received "${lastDestItem}"`);
  }
  if (isXArrayMove) {
    // First perform the add operation.
    if (destinationProcessors) {
      destinationProcessors.splice(destIndex, 0, processor);
    } else {
      const targetProcessor = (0, _utils.getValue)(pathToDestArray.slice(0, -1), state);
      targetProcessor.onFailure = [processor];
    }
    // !! Beyond this point, selectors are no longer usable because we have mutated the data structure!
    // Second, we perform the deletion operation
    sourceProcessors.splice(sourceIndex, 1);

    // If onFailure is empty, delete the array.
    if (!sourceProcessors.length && !sourceProcessor.isRoot) {
      delete sourceProcessor.onFailure;
    }
  } else {
    destinationProcessors.splice(destIndex, 0, processor);
    const targetIdx = sourceIndex > destIndex ? sourceIndex + 1 : sourceIndex;
    sourceProcessors.splice(targetIdx, 1);
  }
  return {
    ...state,
    processors: [...state.processors],
    onFailure: [...state.onFailure]
  };
};
exports.unsafeProcessorMove = unsafeProcessorMove;