"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createLogThresholdExecutor = exports.LogsThresholdFiredActionGroupId = exports.FIRED_ACTIONS = void 0;
exports.executeAlert = executeAlert;
exports.executeRatioAlert = executeRatioAlert;
exports.processUngroupedResults = exports.processUngroupedRatioResults = exports.processGroupByResults = exports.processGroupByRatioResults = exports.getUngroupedESQuery = exports.getGroupedESQuery = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/observability-plugin/common");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common2 = require("@kbn/spaces-plugin/common");
var _std = require("@kbn/std");
var _ecs_field_map = require("@kbn/rule-registry-plugin/common/assets/field_maps/ecs_field_map");
var _get_chart_group_names = require("../../../../common/utils/get_chart_group_names");
var _log_threshold = require("../../../../common/alerting/logs/log_threshold");
var _runtime_types = require("../../../../common/runtime_types");
var _alert_link = require("../../../../common/formatters/alert_link");
var _utils = require("../common/utils");
var _reason_formatters = require("./reason_formatters");
var _query_helpers = require("../../../../common/alerting/logs/log_threshold/query_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// no specific state used
// no specific state used
// no specific instance context used

const COMPOSITE_GROUP_SIZE = 2000;
const checkValueAgainstComparatorMap = {
  [_log_threshold.Comparator.GT]: (a, b) => a > b,
  [_log_threshold.Comparator.GT_OR_EQ]: (a, b) => a >= b,
  [_log_threshold.Comparator.LT]: (a, b) => a < b,
  [_log_threshold.Comparator.LT_OR_EQ]: (a, b) => a <= b
};

// The executor execution roughly follows a pattern of:
// ES Query generation -> fetching of results -> processing of results.
// With forks for group_by vs ungrouped, and ratio vs non-ratio.

const createLogThresholdExecutor = libs => libs.logsRules.createLifecycleRuleExecutor(async ({
  services,
  params,
  spaceId,
  startedAt
}) => {
  const {
    alertFactory: {
      alertLimit
    },
    alertWithLifecycle,
    savedObjectsClient,
    scopedClusterClient,
    getAlertStartedDate,
    getAlertUuid,
    getAlertByAlertUuid
  } = services;
  const {
    basePath,
    alertsLocator
  } = libs;
  const alertFactory = (id, reason, value, threshold, actions, rootLevelContext) => {
    const alertContext = actions != null ? actions.reduce((next, action) => Object.assign(next, action.context), {}) : {};
    const alert = alertWithLifecycle({
      id,
      fields: {
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: threshold,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: value,
        [_ruleDataUtils.ALERT_REASON]: reason,
        [_ruleDataUtils.ALERT_CONTEXT]: alertContext,
        ...(0, _utils.flattenAdditionalContext)(rootLevelContext)
      }
    });
    if (actions && actions.length > 0) {
      var _getAlertStartedDate;
      const indexedStartedAt = (_getAlertStartedDate = getAlertStartedDate(id)) !== null && _getAlertStartedDate !== void 0 ? _getAlertStartedDate : startedAt.toISOString();
      const relativeViewInAppUrl = (0, _alert_link.getLogsAppAlertUrl)(new Date(indexedStartedAt).getTime());
      const viewInAppUrl = (0, _common2.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, relativeViewInAppUrl);
      const sharedContext = {
        timestamp: startedAt.toISOString(),
        viewInAppUrl
      };
      (0, _std.asyncForEach)(actions, async actionSet => {
        const {
          actionGroup,
          context
        } = actionSet;
        const alertInstanceId = context.group || id;
        const alertUuid = getAlertUuid(alertInstanceId);
        alert.scheduleActions(actionGroup, {
          ...sharedContext,
          ...context,
          alertDetailsUrl: (0, _common.getAlertDetailsUrl)(libs.basePath, spaceId, alertUuid)
        });
      });
    }
    alert.replaceState({
      alertState: _log_threshold.AlertStates.ALERT
    });
    return alert;
  };
  const [, {
    logsShared
  }] = await libs.getStartServices();
  try {
    const validatedParams = (0, _runtime_types.decodeOrThrow)(_log_threshold.ruleParamsRT)(params);
    const {
      indices,
      timestampField,
      runtimeMappings
    } = await logsShared.logViews.getClient(savedObjectsClient, scopedClusterClient.asCurrentUser).getResolvedLogView(validatedParams.logView);
    if (!(0, _log_threshold.isRatioRuleParams)(validatedParams)) {
      await executeAlert(validatedParams, timestampField, indices, runtimeMappings, scopedClusterClient.asCurrentUser, alertFactory, alertLimit, startedAt.valueOf());
    } else {
      await executeRatioAlert(validatedParams, timestampField, indices, runtimeMappings, scopedClusterClient.asCurrentUser, alertFactory, alertLimit, startedAt.valueOf());
    }
    const {
      getRecoveredAlerts
    } = services.alertFactory.done();
    const recoveredAlerts = getRecoveredAlerts();
    await processRecoveredAlerts({
      basePath,
      getAlertStartedDate,
      getAlertUuid,
      recoveredAlerts,
      spaceId,
      startedAt,
      validatedParams,
      getAlertByAlertUuid,
      alertsLocator
    });
  } catch (e) {
    throw new Error(e);
  }
  return {
    state: {}
  };
});
exports.createLogThresholdExecutor = createLogThresholdExecutor;
async function executeAlert(ruleParams, timestampField, indexPattern, runtimeMappings, esClient, alertFactory, alertLimit, executionTimestamp) {
  const query = getESQuery(ruleParams, timestampField, indexPattern, runtimeMappings, executionTimestamp);
  if (!query) {
    throw new Error('ES query could not be built from the provided alert params');
  }
  if ((0, _log_threshold.hasGroupBy)(ruleParams)) {
    processGroupByResults(await getGroupedResults(query, esClient), ruleParams, alertFactory, alertLimit);
  } else {
    processUngroupedResults(await getUngroupedResults(query, esClient), ruleParams, alertFactory, alertLimit);
  }
}
async function executeRatioAlert(ruleParams, timestampField, indexPattern, runtimeMappings, esClient, alertFactory, alertLimit, executionTimestamp) {
  // Ratio alert params are separated out into two standard sets of alert params
  const numeratorParams = {
    ...ruleParams,
    criteria: (0, _log_threshold.getNumerator)(ruleParams.criteria)
  };
  const denominatorParams = {
    ...ruleParams,
    criteria: (0, _log_threshold.getDenominator)(ruleParams.criteria)
  };
  const numeratorQuery = getESQuery(numeratorParams, timestampField, indexPattern, runtimeMappings, executionTimestamp);
  const denominatorQuery = getESQuery(denominatorParams, timestampField, indexPattern, runtimeMappings, executionTimestamp);
  if (!numeratorQuery || !denominatorQuery) {
    throw new Error('ES query could not be built from the provided ratio alert params');
  }
  if ((0, _log_threshold.hasGroupBy)(ruleParams)) {
    const [numeratorGroupedResults, denominatorGroupedResults] = await Promise.all([getGroupedResults(numeratorQuery, esClient), getGroupedResults(denominatorQuery, esClient)]);
    processGroupByRatioResults(numeratorGroupedResults, denominatorGroupedResults, ruleParams, alertFactory, alertLimit);
  } else {
    const [numeratorUngroupedResults, denominatorUngroupedResults] = await Promise.all([getUngroupedResults(numeratorQuery, esClient), getUngroupedResults(denominatorQuery, esClient)]);
    processUngroupedRatioResults(numeratorUngroupedResults, denominatorUngroupedResults, ruleParams, alertFactory, alertLimit);
  }
}
const getESQuery = (alertParams, timestampField, indexPattern, runtimeMappings, executionTimestamp) => {
  const executionTimeRange = {
    lte: executionTimestamp
  };
  return (0, _log_threshold.hasGroupBy)(alertParams) ? getGroupedESQuery(alertParams, timestampField, indexPattern, runtimeMappings, executionTimeRange) : getUngroupedESQuery(alertParams, timestampField, indexPattern, runtimeMappings, executionTimeRange);
};
const processUngroupedResults = (results, params, alertFactory, alertLimit) => {
  var _results$aggregations, _results$aggregations2, _results$aggregations3, _additionalContextHit;
  const {
    count,
    criteria,
    timeSize,
    timeUnit
  } = params;
  const documentCount = results.hits.total.value;
  const additionalContextHits = (_results$aggregations = results.aggregations) === null || _results$aggregations === void 0 ? void 0 : (_results$aggregations2 = _results$aggregations.additionalContext) === null || _results$aggregations2 === void 0 ? void 0 : (_results$aggregations3 = _results$aggregations2.hits) === null || _results$aggregations3 === void 0 ? void 0 : _results$aggregations3.hits;
  const additionalContext = formatFields(additionalContextHits === null || additionalContextHits === void 0 ? void 0 : (_additionalContextHit = additionalContextHits[0]) === null || _additionalContextHit === void 0 ? void 0 : _additionalContextHit.fields);
  const reasonMessage = (0, _reason_formatters.getReasonMessageForUngroupedCountAlert)(documentCount, count.value, count.comparator, timeSize, timeUnit);
  if (checkValueAgainstComparatorMap[count.comparator](documentCount, count.value)) {
    const actions = [{
      actionGroup: FIRED_ACTIONS.id,
      context: {
        matchingDocuments: documentCount,
        conditions: createConditionsMessageForCriteria(criteria),
        group: null,
        isRatio: false,
        reason: reasonMessage,
        ...additionalContext
      }
    }];
    alertFactory(_utils.UNGROUPED_FACTORY_KEY, reasonMessage, documentCount, count.value, actions, additionalContext);
    alertLimit.setLimitReached(alertLimit.getValue() <= 1);
  } else {
    alertLimit.setLimitReached(false);
  }
};
exports.processUngroupedResults = processUngroupedResults;
const processUngroupedRatioResults = (numeratorResults, denominatorResults, params, alertFactory, alertLimit) => {
  var _numeratorResults$agg, _numeratorResults$agg2, _numeratorResults$agg3, _additionalContextHit2;
  const {
    count,
    criteria,
    timeSize,
    timeUnit
  } = params;
  const numeratorCount = numeratorResults.hits.total.value;
  const denominatorCount = denominatorResults.hits.total.value;
  const additionalContextHits = (_numeratorResults$agg = numeratorResults.aggregations) === null || _numeratorResults$agg === void 0 ? void 0 : (_numeratorResults$agg2 = _numeratorResults$agg.additionalContext) === null || _numeratorResults$agg2 === void 0 ? void 0 : (_numeratorResults$agg3 = _numeratorResults$agg2.hits) === null || _numeratorResults$agg3 === void 0 ? void 0 : _numeratorResults$agg3.hits;
  const additionalContext = formatFields(additionalContextHits === null || additionalContextHits === void 0 ? void 0 : (_additionalContextHit2 = additionalContextHits[0]) === null || _additionalContextHit2 === void 0 ? void 0 : _additionalContextHit2.fields);
  const ratio = getRatio(numeratorCount, denominatorCount);
  if (ratio !== undefined && checkValueAgainstComparatorMap[count.comparator](ratio, count.value)) {
    const reasonMessage = (0, _reason_formatters.getReasonMessageForUngroupedRatioAlert)(ratio, count.value, count.comparator, timeSize, timeUnit);
    const actions = [{
      actionGroup: FIRED_ACTIONS.id,
      context: {
        ratio,
        numeratorConditions: createConditionsMessageForCriteria((0, _log_threshold.getNumerator)(criteria)),
        denominatorConditions: createConditionsMessageForCriteria((0, _log_threshold.getDenominator)(criteria)),
        group: null,
        isRatio: true,
        reason: reasonMessage,
        ...additionalContext
      }
    }];
    alertFactory(_utils.UNGROUPED_FACTORY_KEY, reasonMessage, ratio, count.value, actions, additionalContext);
    alertLimit.setLimitReached(alertLimit.getValue() <= 1);
  } else {
    alertLimit.setLimitReached(false);
  }
};
exports.processUngroupedRatioResults = processUngroupedRatioResults;
const getRatio = (numerator, denominator) => {
  // We follow the mathematics principle that dividing by 0 isn't possible,
  // and a ratio is therefore undefined (or indeterminate).
  if (numerator === 0 || denominator === 0) return undefined;
  return numerator / denominator;
};
const getReducedGroupByResults = results => {
  const getGroupName = key => (0, _get_chart_group_names.getChartGroupNames)(Object.values(key));
  const reducedGroupByResults = [];
  if ((0, _log_threshold.isOptimizedGroupedSearchQueryResponse)(results)) {
    for (const groupBucket of results) {
      var _groupBucket$addition, _groupBucket$addition2, _additionalContextHit3;
      const groupName = getGroupName(groupBucket.key);
      const additionalContextHits = (_groupBucket$addition = groupBucket.additionalContext) === null || _groupBucket$addition === void 0 ? void 0 : (_groupBucket$addition2 = _groupBucket$addition.hits) === null || _groupBucket$addition2 === void 0 ? void 0 : _groupBucket$addition2.hits;
      reducedGroupByResults.push({
        name: groupName,
        documentCount: groupBucket.doc_count,
        context: formatFields(additionalContextHits === null || additionalContextHits === void 0 ? void 0 : (_additionalContextHit3 = additionalContextHits[0]) === null || _additionalContextHit3 === void 0 ? void 0 : _additionalContextHit3.fields)
      });
    }
  } else {
    for (const groupBucket of results) {
      var _groupBucket$filtered, _groupBucket$filtered2, _additionalContextHit4;
      const groupName = getGroupName(groupBucket.key);
      const additionalContextHits = (_groupBucket$filtered = groupBucket.filtered_results.additionalContext) === null || _groupBucket$filtered === void 0 ? void 0 : (_groupBucket$filtered2 = _groupBucket$filtered.hits) === null || _groupBucket$filtered2 === void 0 ? void 0 : _groupBucket$filtered2.hits;
      reducedGroupByResults.push({
        name: groupName,
        documentCount: groupBucket.filtered_results.doc_count,
        context: formatFields(additionalContextHits === null || additionalContextHits === void 0 ? void 0 : (_additionalContextHit4 = additionalContextHits[0]) === null || _additionalContextHit4 === void 0 ? void 0 : _additionalContextHit4.fields)
      });
    }
  }
  return reducedGroupByResults;
};
const processGroupByResults = (results, params, alertFactory, alertLimit) => {
  const {
    count,
    criteria,
    timeSize,
    timeUnit
  } = params;
  const groupResults = getReducedGroupByResults(results);
  let remainingAlertCount = alertLimit.getValue();
  for (const group of groupResults) {
    if (remainingAlertCount <= 0) {
      break;
    }
    const documentCount = group.documentCount;
    if (checkValueAgainstComparatorMap[count.comparator](documentCount, count.value)) {
      remainingAlertCount -= 1;
      const reasonMessage = (0, _reason_formatters.getReasonMessageForGroupedCountAlert)(documentCount, count.value, count.comparator, group.name, timeSize, timeUnit);
      const groupByKeysObjectMapping = (0, _utils.getGroupByObject)(params.groupBy, new Set(groupResults.map(groupResult => groupResult.name)));
      const actions = [{
        actionGroup: FIRED_ACTIONS.id,
        context: {
          matchingDocuments: documentCount,
          conditions: createConditionsMessageForCriteria(criteria),
          group: group.name,
          groupByKeys: groupByKeysObjectMapping[group.name],
          isRatio: false,
          reason: reasonMessage,
          ...group.context
        }
      }];
      alertFactory(group.name, reasonMessage, documentCount, count.value, actions, group.context);
    }
  }
  alertLimit.setLimitReached(remainingAlertCount <= 0);
};
exports.processGroupByResults = processGroupByResults;
const processGroupByRatioResults = (numeratorResults, denominatorResults, params, alertFactory, alertLimit) => {
  const {
    count,
    criteria,
    timeSize,
    timeUnit
  } = params;
  const numeratorGroupResults = getReducedGroupByResults(numeratorResults);
  const denominatorGroupResults = getReducedGroupByResults(denominatorResults);
  let remainingAlertCount = alertLimit.getValue();
  for (const numeratorGroup of numeratorGroupResults) {
    if (remainingAlertCount <= 0) {
      break;
    }
    const numeratorDocumentCount = numeratorGroup.documentCount;
    const denominatorGroup = denominatorGroupResults.find(_group => _group.name === numeratorGroup.name);
    // If there is no matching group, a ratio cannot be determined, and is therefore undefined.
    const ratio = denominatorGroup ? getRatio(numeratorDocumentCount, denominatorGroup.documentCount) : undefined;
    if (ratio !== undefined && checkValueAgainstComparatorMap[count.comparator](ratio, count.value)) {
      remainingAlertCount -= 1;
      const reasonMessage = (0, _reason_formatters.getReasonMessageForGroupedRatioAlert)(ratio, count.value, count.comparator, numeratorGroup.name, timeSize, timeUnit);
      const groupByKeysObjectMapping = (0, _utils.getGroupByObject)(params.groupBy, new Set(numeratorGroupResults.map(groupResult => groupResult.name)));
      const actions = [{
        actionGroup: FIRED_ACTIONS.id,
        context: {
          ratio,
          numeratorConditions: createConditionsMessageForCriteria((0, _log_threshold.getNumerator)(criteria)),
          denominatorConditions: createConditionsMessageForCriteria((0, _log_threshold.getDenominator)(criteria)),
          group: numeratorGroup.name,
          groupByKeys: groupByKeysObjectMapping[numeratorGroup.name],
          isRatio: true,
          reason: reasonMessage,
          ...numeratorGroup.context
        }
      }];
      alertFactory(numeratorGroup.name, reasonMessage, ratio, count.value, actions, numeratorGroup.context);
    }
  }
  alertLimit.setLimitReached(remainingAlertCount <= 0);
};
exports.processGroupByRatioResults = processGroupByRatioResults;
const getGroupedESQuery = (params, timestampField, index, runtimeMappings, executionTimeRange) => {
  // IMPORTANT:
  // For the group by scenario we need to account for users utilizing "less than" configurations
  // to attempt to match on "0", e.g. something has stopped reporting. We need to cast a wider net for these
  // configurations to try and capture more documents, so that the filtering doesn't make the group "disappear".
  // Due to this there are two forks in the group by code, one where we can optimize the filtering early, and one where
  // it is an inner aggregation. "Less than" configurations with high cardinality group by fields can cause severe performance
  // problems.

  const {
    groupBy,
    count: {
      comparator,
      value
    }
  } = params;
  if (!groupBy || !groupBy.length) {
    return;
  }
  const {
    rangeFilter,
    groupedRangeFilter,
    mustFilters,
    mustNotFilters
  } = (0, _query_helpers.buildFiltersFromCriteria)(params, timestampField, executionTimeRange);
  if ((0, _log_threshold.isOptimizableGroupedThreshold)(comparator, value)) {
    const aggregations = {
      groups: {
        composite: {
          size: COMPOSITE_GROUP_SIZE,
          sources: groupBy.map((field, groupIndex) => ({
            [`group-${groupIndex}-${field}`]: {
              terms: {
                field
              }
            }
          }))
        },
        aggregations: {
          ...getContextAggregation(params)
        }
      }
    };
    const body = {
      query: {
        bool: {
          filter: [rangeFilter, ...mustFilters],
          ...(mustNotFilters.length > 0 && {
            must_not: mustNotFilters
          })
        }
      },
      aggregations,
      runtime_mappings: runtimeMappings,
      size: 0
    };
    return {
      index,
      allow_no_indices: true,
      ignore_unavailable: true,
      body
    };
  } else {
    const aggregations = {
      groups: {
        composite: {
          size: COMPOSITE_GROUP_SIZE,
          sources: groupBy.map((field, groupIndex) => ({
            [`group-${groupIndex}-${field}`]: {
              terms: {
                field
              }
            }
          }))
        },
        aggregations: {
          filtered_results: {
            filter: {
              bool: {
                // Scope the inner filtering back to the unpadded range
                filter: [rangeFilter, ...mustFilters],
                ...(mustNotFilters.length > 0 && {
                  must_not: mustNotFilters
                })
              }
            },
            aggregations: {
              ...getContextAggregation(params)
            }
          }
        }
      }
    };
    const body = {
      query: {
        bool: {
          filter: [groupedRangeFilter]
        }
      },
      aggregations,
      runtime_mappings: runtimeMappings,
      size: 0
    };
    return {
      index,
      allow_no_indices: true,
      ignore_unavailable: true,
      body
    };
  }
};
exports.getGroupedESQuery = getGroupedESQuery;
const getUngroupedESQuery = (params, timestampField, index, runtimeMappings, executionTimeRange) => {
  const {
    rangeFilter,
    mustFilters,
    mustNotFilters
  } = (0, _query_helpers.buildFiltersFromCriteria)(params, timestampField, executionTimeRange);
  const body = {
    // Ensure we accurately track the hit count for the ungrouped case, otherwise we can only ensure accuracy up to 10,000.
    track_total_hits: true,
    query: {
      bool: {
        filter: [rangeFilter, ...mustFilters],
        ...(mustNotFilters.length > 0 && {
          must_not: mustNotFilters
        })
      }
    },
    aggregations: {
      ...getContextAggregation(params)
    },
    runtime_mappings: runtimeMappings,
    size: 0
  };
  return {
    index,
    allow_no_indices: true,
    ignore_unavailable: true,
    body
  };
};
exports.getUngroupedESQuery = getUngroupedESQuery;
const getUngroupedResults = async (query, esClient) => {
  return (0, _runtime_types.decodeOrThrow)(_log_threshold.UngroupedSearchQueryResponseRT)(await esClient.search(query));
};
const getGroupedResults = async (query, esClient) => {
  let compositeGroupBuckets = [];
  let lastAfterKey;
  while (true) {
    const queryWithAfterKey = {
      ...query
    };
    queryWithAfterKey.body.aggregations.groups.composite.after = lastAfterKey;
    const groupResponse = (0, _runtime_types.decodeOrThrow)(_log_threshold.GroupedSearchQueryResponseRT)(await esClient.search(queryWithAfterKey));
    compositeGroupBuckets = [...compositeGroupBuckets, ...groupResponse.aggregations.groups.buckets];
    lastAfterKey = groupResponse.aggregations.groups.after_key;
    if (groupResponse.aggregations.groups.buckets.length < COMPOSITE_GROUP_SIZE) {
      break;
    }
  }
  return compositeGroupBuckets;
};
const processRecoveredAlerts = async ({
  basePath,
  getAlertStartedDate,
  getAlertUuid,
  recoveredAlerts,
  spaceId,
  startedAt,
  validatedParams,
  getAlertByAlertUuid,
  alertsLocator
}) => {
  const groupByKeysObjectForRecovered = (0, _utils.getGroupByObject)(validatedParams.groupBy, new Set(recoveredAlerts.map(recoveredAlert => recoveredAlert.getId())));
  for (const alert of recoveredAlerts) {
    var _getAlertStartedDate2;
    const recoveredAlertId = alert.getId();
    const indexedStartedAt = (_getAlertStartedDate2 = getAlertStartedDate(recoveredAlertId)) !== null && _getAlertStartedDate2 !== void 0 ? _getAlertStartedDate2 : startedAt.toISOString();
    const relativeViewInAppUrl = (0, _alert_link.getLogsAppAlertUrl)(new Date(indexedStartedAt).getTime());
    const alertUuid = getAlertUuid(recoveredAlertId);
    const alertHits = alertUuid ? await getAlertByAlertUuid(alertUuid) : undefined;
    const additionalContext = (0, _utils.getContextForRecoveredAlerts)(alertHits);
    const viewInAppUrl = (0, _common2.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, relativeViewInAppUrl);
    const baseContext = {
      alertDetailsUrl: (0, _common.getAlertDetailsUrl)(basePath, spaceId, alertUuid),
      group: (0, _log_threshold.hasGroupBy)(validatedParams) ? recoveredAlertId : null,
      groupByKeys: groupByKeysObjectForRecovered[recoveredAlertId],
      timestamp: startedAt.toISOString(),
      viewInAppUrl,
      ...additionalContext
    };
    if ((0, _log_threshold.isRatioRuleParams)(validatedParams)) {
      const {
        criteria
      } = validatedParams;
      alert.setContext({
        ...baseContext,
        numeratorConditions: createConditionsMessageForCriteria((0, _log_threshold.getNumerator)(criteria)),
        denominatorConditions: createConditionsMessageForCriteria((0, _log_threshold.getDenominator)(criteria)),
        isRatio: true
      });
    } else {
      const {
        criteria
      } = validatedParams;
      alert.setContext({
        ...baseContext,
        conditions: createConditionsMessageForCriteria(criteria),
        isRatio: false
      });
    }
  }
};
const createConditionsMessageForCriteria = criteria => criteria.map(criterion => {
  const {
    field,
    comparator,
    value
  } = criterion;
  return `${field} ${comparator} ${value}`;
}).join(' and ');

// When the Alerting plugin implements support for multiple action groups, add additional
// action groups here to send different messages, e.g. a recovery notification
const LogsThresholdFiredActionGroupId = 'logs.threshold.fired';
exports.LogsThresholdFiredActionGroupId = LogsThresholdFiredActionGroupId;
const FIRED_ACTIONS = {
  id: LogsThresholdFiredActionGroupId,
  name: _i18n.i18n.translate('xpack.infra.logs.alerting.threshold.fired', {
    defaultMessage: 'Fired'
  })
};
exports.FIRED_ACTIONS = FIRED_ACTIONS;
const getContextAggregation = params => {
  const validPrefixForContext = ['host', 'cloud', 'orchestrator', 'container', 'labels', 'tags'];
  const positiveCriteria = params.criteria.filter(criterion => _query_helpers.positiveComparators.includes(criterion.comparator));
  const fieldsFromGroupBy = params.groupBy ? getFieldsSet(params.groupBy, validPrefixForContext) : new Set();
  const fieldsFromCriteria = getFieldsSet(positiveCriteria.map(criterion => criterion.field), validPrefixForContext);
  const fieldsPrefixList = Array.from(new Set([...fieldsFromGroupBy, ...fieldsFromCriteria]));
  const fieldsList = fieldsPrefixList.map(prefix => prefix === 'tags' ? prefix : `${prefix}.*`);
  const additionalContextAgg = fieldsList.length > 0 ? {
    additionalContext: {
      top_hits: {
        size: 1,
        fields: fieldsList,
        _source: false
      }
    }
  } : null;
  return additionalContextAgg;
};
const getFieldsSet = (groupBy, validPrefix) => {
  return new Set(groupBy === null || groupBy === void 0 ? void 0 : groupBy.map(currentGroupBy => currentGroupBy.split('.')[0]).filter(groupByPrefix => validPrefix.includes(groupByPrefix)));
};
const fieldsToExclude = ['disk', 'network', 'cpu', 'memory'];
const formatFields = contextFields => {
  if (!contextFields) return undefined;
  const formattedContextFields = {};
  Object.entries(contextFields).forEach(([key, value]) => {
    if (key in _ecs_field_map.ecsFieldMap && !excludeField(key)) {
      formattedContextFields[key] = _ecs_field_map.ecsFieldMap[key].array ? value : value === null || value === void 0 ? void 0 : value[0];
    }
  });
  return (0, _utils.unflattenObject)(formattedContextFields);
};
const excludeField = key => {
  var _key$split;
  return fieldsToExclude.includes((_key$split = key.split('.')) === null || _key$split === void 0 ? void 0 : _key$split[1]);
};