"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.putOutputHandler = exports.postOutputHandler = exports.postLogstashApiKeyHandler = exports.getOutputsHandler = exports.getOneOuputHandler = exports.getLatestOutputHealth = exports.deleteOutputHandler = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _constants = require("../../../common/constants");
var _output = require("../../services/output");
var _errors = require("../../errors");
var _services = require("../../services");
var _api_keys = require("../../services/api_keys");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function ensureNoDuplicateSecrets(output) {
  var _output$secrets, _output$ssl, _output$secrets2, _output$secrets2$ssl, _output$secrets3;
  if (output.type === _constants.outputType.Kafka && output !== null && output !== void 0 && output.password && output !== null && output !== void 0 && (_output$secrets = output.secrets) !== null && _output$secrets !== void 0 && _output$secrets.password) {
    throw _boom.default.badRequest('Cannot specify both password and secrets.password');
  }
  if ((output.type === _constants.outputType.Kafka || output.type === _constants.outputType.Logstash) && (_output$ssl = output.ssl) !== null && _output$ssl !== void 0 && _output$ssl.key && (_output$secrets2 = output.secrets) !== null && _output$secrets2 !== void 0 && (_output$secrets2$ssl = _output$secrets2.ssl) !== null && _output$secrets2$ssl !== void 0 && _output$secrets2$ssl.key) {
    throw _boom.default.badRequest('Cannot specify both ssl.key and secrets.ssl.key');
  }
  if (output.type === _constants.outputType.RemoteElasticsearch && output.service_token && (_output$secrets3 = output.secrets) !== null && _output$secrets3 !== void 0 && _output$secrets3.service_token) {
    throw _boom.default.badRequest('Cannot specify both service_token and secrets.service_token');
  }
}
const getOutputsHandler = async (context, request, response) => {
  const soClient = (await context.core).savedObjects.client;
  try {
    const outputs = await _output.outputService.list(soClient);
    const body = {
      items: outputs.items,
      page: outputs.page,
      perPage: outputs.perPage,
      total: outputs.total
    };
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getOutputsHandler = getOutputsHandler;
const getOneOuputHandler = async (context, request, response) => {
  const soClient = (await context.core).savedObjects.client;
  try {
    const output = await _output.outputService.get(soClient, request.params.outputId);
    const body = {
      item: output
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `Output ${request.params.outputId} not found`
        }
      });
    }
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getOneOuputHandler = getOneOuputHandler;
const putOutputHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const outputUpdate = request.body;
  try {
    validateOutputServerless(outputUpdate.type);
    ensureNoDuplicateSecrets(outputUpdate);
    await _output.outputService.update(soClient, esClient, request.params.outputId, outputUpdate);
    const output = await _output.outputService.get(soClient, request.params.outputId);
    if (output.is_default || output.is_default_monitoring) {
      await _services.agentPolicyService.bumpAllAgentPolicies(soClient, esClient);
    } else {
      await _services.agentPolicyService.bumpAllAgentPoliciesForOutput(soClient, esClient, output.id);
    }
    const body = {
      item: output
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `Output ${request.params.outputId} not found`
        }
      });
    }
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.putOutputHandler = putOutputHandler;
const postOutputHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  try {
    const {
      id,
      ...newOutput
    } = request.body;
    validateOutputServerless(newOutput.type);
    ensureNoDuplicateSecrets(newOutput);
    const output = await _output.outputService.create(soClient, esClient, newOutput, {
      id
    });
    if (output.is_default || output.is_default_monitoring) {
      await _services.agentPolicyService.bumpAllAgentPolicies(soClient, esClient);
    }
    const body = {
      item: output
    };
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.postOutputHandler = postOutputHandler;
function validateOutputServerless(type) {
  const cloudSetup = _services.appContextService.getCloud();
  if (cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled && type === _constants.outputType.RemoteElasticsearch) {
    throw _boom.default.badRequest('Output type remote_elasticsearch not supported in serverless');
  }
}
const deleteOutputHandler = async (context, request, response) => {
  const soClient = (await context.core).savedObjects.client;
  try {
    await _output.outputService.delete(soClient, request.params.outputId);
    const body = {
      id: request.params.outputId
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `Output ${request.params.outputId} not found`
        }
      });
    }
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.deleteOutputHandler = deleteOutputHandler;
const postLogstashApiKeyHandler = async (context, request, response) => {
  const esClient = (await context.core).elasticsearch.client.asCurrentUser;
  try {
    const hasCreatePrivileges = await (0, _api_keys.canCreateLogstashApiKey)(esClient);
    if (!hasCreatePrivileges) {
      throw new _errors.FleetUnauthorizedError('Missing permissions to create logstash API key');
    }
    const apiKey = await (0, _api_keys.generateLogstashApiKey)(esClient);
    const body = {
      // Logstash expect the key to be formatted like this id:key
      api_key: `${apiKey.id}:${apiKey.api_key}`
    };
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.postLogstashApiKeyHandler = postLogstashApiKeyHandler;
const getLatestOutputHealth = async (context, request, response) => {
  const esClient = (await context.core).elasticsearch.client.asCurrentUser;
  try {
    const outputHealth = await _output.outputService.getLatestOutputHealth(esClient, request.params.outputId);
    return response.ok({
      body: outputHealth
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getLatestOutputHealth = getLatestOutputHealth;