"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useActionStatus = useActionStatus;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _hooks = require("../../../../hooks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useActionStatus(onAbortSuccess, refreshAgentActivity) {
  const [currentActions, setCurrentActions] = (0, _react.useState)([]);
  const [isFirstLoading, setIsFirstLoading] = (0, _react.useState)(true);
  const {
    notifications,
    overlays
  } = (0, _hooks.useStartServices)();
  const refreshActions = (0, _react.useCallback)(async () => {
    try {
      const res = await (0, _hooks.sendGetActionStatus)();
      setIsFirstLoading(false);
      if (res.error) {
        throw res.error;
      }
      if (!res.data) {
        throw new Error('No data');
      }
      setCurrentActions(res.data.items);
    } catch (err) {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.fleet.actionStatus.fetchRequestError', {
          defaultMessage: 'An error happened while fetching action status'
        })
      });
    }
  }, [notifications.toasts]);
  if (isFirstLoading) {
    refreshActions();
  }
  (0, _react.useEffect)(() => {
    if (refreshAgentActivity) {
      refreshActions();
    }
    return () => {
      setCurrentActions([]);
      setIsFirstLoading(true);
    };
  }, [refreshActions, refreshAgentActivity]);
  const abortUpgrade = (0, _react.useCallback)(async action => {
    try {
      const confirmRes = await overlays.openConfirm(_i18n.i18n.translate('xpack.fleet.currentUpgrade.confirmDescription', {
        defaultMessage: 'This action will cancel upgrade of {nbAgents, plural, one {# agent} other {# agents}}',
        values: {
          nbAgents: action.nbAgentsActioned - action.nbAgentsAck
        }
      }), {
        title: _i18n.i18n.translate('xpack.fleet.currentUpgrade.confirmTitle', {
          defaultMessage: 'Cancel upgrade?'
        })
      });
      if (!confirmRes) {
        return;
      }
      await (0, _hooks.sendPostCancelAction)(action.actionId);
      await Promise.all([refreshActions(), onAbortSuccess()]);
    } catch (err) {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.fleet.currentUpgrade.abortRequestError', {
          defaultMessage: 'An error happened while cancelling upgrade'
        })
      });
    }
  }, [refreshActions, notifications.toasts, overlays, onAbortSuccess]);
  return {
    currentActions,
    refreshActions,
    abortUpgrade,
    isFirstLoading
  };
}