"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useQuickStartCreateForm = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _hooks = require("../../../hooks");
var _use_service_token = require("../../../hooks/use_service_token");
var _use_select_fleet_server_policy = require("./use_select_fleet_server_policy");
var _use_fleet_server_host = require("./use_fleet_server_host");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const QUICK_START_FLEET_SERVER_POLICY_FIELDS = {
  id: 'fleet-server-policy',
  name: 'Fleet Server Policy',
  description: 'Fleet Server policy generated by Kibana',
  namespace: 'default',
  has_fleet_server: true,
  monitoring_enabled: ['logs', 'metrics'],
  is_default_fleet_server: true
};
/**
 * Provides a unified interface that combines the following operations:
 * 1. Setting a Fleet Server host in Fleet's settings
 * 2. Creating an agent policy that contains the `fleet_server` integration
 * 3. Generating a service token used by Fleet Server
 */
const useQuickStartCreateForm = () => {
  const [status, setStatus] = (0, _react.useState)('initial');
  const [error, setError] = (0, _react.useState)();
  const {
    fleetServerHosts,
    fleetServerHost,
    isFleetServerHostSubmitted,
    saveFleetServerHost,
    error: fleetServerError,
    setFleetServerHost,
    validate,
    inputs
  } = (0, _use_fleet_server_host.useFleetServerHost)();

  // When a validation error is surfaced from the Fleet Server host form, we want to treat it
  // the same way we do errors from the service token or policy creation steps
  (0, _react.useEffect)(() => {
    setStatus('error');
    setError(fleetServerError);
  }, [fleetServerError]);
  const {
    notifications
  } = (0, _hooks.useStartServices)();
  const {
    fleetServerPolicyId,
    setFleetServerPolicyId
  } = (0, _use_select_fleet_server_policy.useSelectFleetServerPolicy)();
  const {
    serviceToken,
    generateServiceToken
  } = (0, _use_service_token.useServiceToken)();
  const submit = (0, _react.useCallback)(async () => {
    try {
      if (!fleetServerHost && validate() || fleetServerHost) {
        var _existingPolicy$data;
        setStatus('loading');
        const newFleetServerHost = {
          name: inputs.nameInput.value,
          host_urls: inputs.hostUrlsInput.value,
          is_default: inputs.isDefaultInput.value,
          is_preconfigured: false
        };
        if (!fleetServerHost) {
          const res = await saveFleetServerHost(newFleetServerHost);
          setFleetServerHost(res);
        }
        await generateServiceToken();
        const existingPolicy = await (0, _hooks.sendGetOneAgentPolicy)(QUICK_START_FLEET_SERVER_POLICY_FIELDS.id);

        // Don't attempt to create the policy if it's already been created in a previous quick start flow
        if ((_existingPolicy$data = existingPolicy.data) !== null && _existingPolicy$data !== void 0 && _existingPolicy$data.item) {
          var _existingPolicy$data2;
          setFleetServerPolicyId((_existingPolicy$data2 = existingPolicy.data) === null || _existingPolicy$data2 === void 0 ? void 0 : _existingPolicy$data2.item.id);
        } else {
          var _createPolicyResponse;
          const createPolicyResponse = await (0, _hooks.sendCreateAgentPolicy)(QUICK_START_FLEET_SERVER_POLICY_FIELDS, {
            withSysMonitoring: true
          });
          setFleetServerPolicyId((_createPolicyResponse = createPolicyResponse.data) === null || _createPolicyResponse === void 0 ? void 0 : _createPolicyResponse.item.id);
        }
        setStatus('success');
      }
    } catch (err) {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.fleet.fleetServerSetup.errorAddingFleetServerHostTitle', {
          defaultMessage: 'Error adding Fleet Server host'
        })
      });
      setStatus('error');
      setError(err.message);
    }
  }, [validate, fleetServerHost, inputs.nameInput.value, inputs.hostUrlsInput.value, inputs.isDefaultInput.value, setFleetServerHost, saveFleetServerHost, generateServiceToken, setFleetServerPolicyId, notifications.toasts]);
  return {
    status,
    error,
    submit,
    fleetServerPolicyId,
    fleetServerHosts,
    fleetServerHost,
    setFleetServerHost,
    isFleetServerHostSubmitted,
    serviceToken,
    inputs
  };
};
exports.useQuickStartCreateForm = useQuickStartCreateForm;