"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateAuthz = void 0;
exports.calculateEndpointExceptionsPrivilegesFromCapabilities = calculateEndpointExceptionsPrivilegesFromCapabilities;
exports.calculateEndpointExceptionsPrivilegesFromKibanaPrivileges = calculateEndpointExceptionsPrivilegesFromKibanaPrivileges;
exports.calculatePackagePrivilegesFromCapabilities = calculatePackagePrivilegesFromCapabilities;
exports.calculatePackagePrivilegesFromKibanaPrivileges = calculatePackagePrivilegesFromKibanaPrivileges;
exports.getAuthorizationFromPrivileges = getAuthorizationFromPrivileges;
var _plugin = require("./constants/plugin");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const calculateAuthz = ({
  fleet,
  integrations,
  isSuperuser
}) => ({
  fleet: {
    all: fleet.all && (integrations.all || integrations.read),
    // These are currently used by Fleet Server setup
    setup: fleet.all || fleet.setup,
    readEnrollmentTokens: fleet.all || fleet.setup,
    readAgentPolicies: fleet.all || fleet.setup
  },
  integrations: {
    readPackageInfo: fleet.all || fleet.setup || integrations.all || integrations.read,
    readInstalledPackages: integrations.all || integrations.read,
    installPackages: fleet.all && integrations.all,
    upgradePackages: fleet.all && integrations.all,
    removePackages: fleet.all && integrations.all,
    uploadPackages: isSuperuser,
    readPackageSettings: fleet.all && integrations.all,
    writePackageSettings: fleet.all && integrations.all,
    readIntegrationPolicies: fleet.all && (integrations.all || integrations.read),
    writeIntegrationPolicies: fleet.all && integrations.all
  }
});
exports.calculateAuthz = calculateAuthz;
function calculatePackagePrivilegesFromCapabilities(capabilities) {
  if (!capabilities) {
    return {};
  }
  const endpointActions = Object.entries(_constants.ENDPOINT_PRIVILEGES).reduce((acc, [privilege, {
    privilegeName
  }]) => {
    acc[privilege] = {
      executePackageAction: capabilities.siem && capabilities.siem[privilegeName] || false
    };
    return acc;
  }, {});
  const transformActions = Object.keys(capabilities.transform).reduce((acc, privilegeName) => {
    acc[privilegeName] = {
      executePackageAction: capabilities.transform[privilegeName] || false
    };
    return acc;
  }, {});
  return {
    endpoint: {
      actions: endpointActions
    },
    transform: {
      actions: transformActions
    }
  };
}
function calculateEndpointExceptionsPrivilegesFromCapabilities(capabilities) {
  if (!capabilities || !capabilities.siem) {
    return;
  }
  const endpointExceptionsActions = Object.keys(_constants.ENDPOINT_EXCEPTIONS_PRIVILEGES).reduce((acc, privilegeName) => {
    acc[privilegeName] = capabilities.siem[privilegeName] || false;
    return acc;
  }, {});
  return {
    actions: endpointExceptionsActions
  };
}
function getAuthorizationFromPrivileges({
  kibanaPrivileges,
  searchPrivilege = '',
  prefix = ''
}) {
  const privilege = kibanaPrivileges.find(p => {
    if (prefix.length && searchPrivilege.length) {
      return p.privilege.endsWith(`${prefix}${searchPrivilege}`);
    } else if (prefix.length) {
      return p.privilege.endsWith(`${prefix}`);
    } else if (searchPrivilege.length) {
      return p.privilege.endsWith(`${searchPrivilege}`);
    }
    return false;
  });
  return !!(privilege !== null && privilege !== void 0 && privilege.authorized);
}
function calculatePackagePrivilegesFromKibanaPrivileges(kibanaPrivileges) {
  if (!kibanaPrivileges || !kibanaPrivileges.length) {
    return {};
  }
  const endpointActions = Object.entries(_constants.ENDPOINT_PRIVILEGES).reduce((acc, [privilege, {
    appId,
    privilegeSplit,
    privilegeName
  }]) => {
    const kibanaPrivilege = getAuthorizationFromPrivileges({
      kibanaPrivileges,
      prefix: `${appId}${privilegeSplit}`,
      searchPrivilege: privilegeName
    });
    acc[privilege] = {
      executePackageAction: kibanaPrivilege
    };
    return acc;
  }, {});
  const hasTransformAdmin = getAuthorizationFromPrivileges({
    kibanaPrivileges,
    prefix: `${_plugin.TRANSFORM_PLUGIN_ID}-`,
    searchPrivilege: `admin`
  });
  const transformActions = {
    canCreateTransform: {
      executePackageAction: hasTransformAdmin
    },
    canDeleteTransform: {
      executePackageAction: hasTransformAdmin
    },
    canStartStopTransform: {
      executePackageAction: hasTransformAdmin
    },
    canGetTransform: {
      executePackageAction: getAuthorizationFromPrivileges({
        kibanaPrivileges,
        prefix: `${_plugin.TRANSFORM_PLUGIN_ID}-`,
        searchPrivilege: `read`
      })
    }
  };
  return {
    endpoint: {
      actions: endpointActions
    },
    transform: {
      actions: transformActions
    }
  };
}
function calculateEndpointExceptionsPrivilegesFromKibanaPrivileges(kibanaPrivileges) {
  if (!kibanaPrivileges || !kibanaPrivileges.length) {
    return;
  }
  const endpointExceptionsActions = Object.entries(_constants.ENDPOINT_EXCEPTIONS_PRIVILEGES).reduce((acc, [privilege, {
    appId,
    privilegeSplit,
    privilegeName
  }]) => {
    acc[privilege] = getAuthorizationFromPrivileges({
      kibanaPrivileges,
      searchPrivilege: privilegeName
    });
    return acc;
  }, {});
  return {
    actions: endpointExceptionsActions
  };
}