"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchIndexData = exports.getIndexDataMapper = exports.getIndexData = void 0;
var _map_index_stats = require("./map_index_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSearchIndexData = async (client, indexPattern, expandWildcards, returnHiddenIndices, includeAliases, alwaysShowPattern) => {
  const totalIndices = await client.asCurrentUser.indices.get({
    expand_wildcards: expandWildcards,
    // for better performance only compute aliases and settings of indices but not mappings
    features: ['aliases', 'settings'],
    // only get specified index properties from ES to keep the response under 536MB
    // node.js string length limit: https://github.com/nodejs/node/issues/33960
    filter_path: ['*.aliases', '*.settings.index.hidden'],
    index: indexPattern
  });

  // Index names that with one of their aliases match with the alwaysShowPattern
  const alwaysShowPatternMatches = new Set();
  const indexAndAliasNames = Object.keys(totalIndices).reduce((accum, indexName) => {
    accum.push(indexName);
    if (includeAliases) {
      const aliases = Object.keys(totalIndices[indexName].aliases);
      aliases.forEach(alias => {
        accum.push(alias);

        // Add indexName to the set if an alias matches the pattern
        if (alwaysShowPattern !== null && alwaysShowPattern !== void 0 && alwaysShowPattern.alias_pattern && alias.startsWith(alwaysShowPattern === null || alwaysShowPattern === void 0 ? void 0 : alwaysShowPattern.alias_pattern)) {
          alwaysShowPatternMatches.add(indexName);
        }
      });
    }
    return accum;
  }, []);
  const indicesNames = returnHiddenIndices ? Object.keys(totalIndices) : Object.keys(totalIndices).filter(indexName => !isHidden(totalIndices[indexName]) || (alwaysShowPattern === null || alwaysShowPattern === void 0 ? void 0 : alwaysShowPattern.index_pattern) && indexName.startsWith(alwaysShowPattern.index_pattern));
  return {
    allIndexMatches: totalIndices,
    alwaysShowMatchNames: Array.from(alwaysShowPatternMatches),
    expandWildcards,
    indexAndAliasNames,
    indicesNames
  };
};
exports.getSearchIndexData = getSearchIndexData;
const getIndexDataMapper = totalIndexData => {
  return indexName => (0, _map_index_stats.mapIndexStats)(totalIndexData.allIndexMatches[indexName], totalIndexData.indicesStats[indexName], indexName);
};
exports.getIndexDataMapper = getIndexDataMapper;
function isHidden(index) {
  var _index$settings, _index$settings$index, _index$settings2, _index$settings2$inde;
  return ((_index$settings = index.settings) === null || _index$settings === void 0 ? void 0 : (_index$settings$index = _index$settings.index) === null || _index$settings$index === void 0 ? void 0 : _index$settings$index.hidden) === true || ((_index$settings2 = index.settings) === null || _index$settings2 === void 0 ? void 0 : (_index$settings2$inde = _index$settings2.index) === null || _index$settings2$inde === void 0 ? void 0 : _index$settings2$inde.hidden) === 'true';
}
function isClosed(index) {
  var _index$settings3, _index$settings3$inde, _index$settings4, _index$settings4$inde;
  return ((_index$settings3 = index.settings) === null || _index$settings3 === void 0 ? void 0 : (_index$settings3$inde = _index$settings3.index) === null || _index$settings3$inde === void 0 ? void 0 : _index$settings3$inde.verified_before_close) === true || ((_index$settings4 = index.settings) === null || _index$settings4 === void 0 ? void 0 : (_index$settings4$inde = _index$settings4.index) === null || _index$settings4$inde === void 0 ? void 0 : _index$settings4$inde.verified_before_close) === 'true';
}
const getIndexData = async (client, onlyShowSearchOptimizedIndices, returnHiddenIndices, searchQuery) => {
  const expandWildcards = returnHiddenIndices ? ['hidden', 'all'] : ['open'];
  const indexPattern = searchQuery ? `*${searchQuery}*` : '*';
  const allIndexMatches = await client.asCurrentUser.indices.get({
    expand_wildcards: expandWildcards,
    // for better performance only compute aliases and settings of indices but not mappings
    features: ['aliases', 'settings'],
    // only get specified index properties from ES to keep the response under 536MB
    // node.js string length limit: https://github.com/nodejs/node/issues/33960
    filter_path: ['*.aliases', '*.settings.index.hidden', '*.settings.index.verified_before_close'],
    index: onlyShowSearchOptimizedIndices ? 'search-*' : indexPattern
  });
  const allIndexNames = returnHiddenIndices ? Object.keys(allIndexMatches).filter(indexName => allIndexMatches[indexName] && !isClosed(allIndexMatches[indexName])) : Object.keys(allIndexMatches).filter(indexName => allIndexMatches[indexName] && !isHidden(allIndexMatches[indexName]) && !isClosed(allIndexMatches[indexName]));
  const indexNames = onlyShowSearchOptimizedIndices && searchQuery ? allIndexNames.filter(indexName => indexName.includes(searchQuery.toLowerCase())) : allIndexNames;
  return {
    indexData: allIndexMatches,
    indexNames
  };
};
exports.getIndexData = getIndexData;