"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.wait = exports.parseErrorMessage = exports.getMessageFromLangChainResponse = exports.getFormattedMessageContent = exports.callAgentWithRetry = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const wait = ms => new Promise(resolve => setTimeout(resolve, ms));
exports.wait = wait;
const callAgentWithRetry = async ({
  agent,
  exampleId,
  messages,
  logger,
  maxRetries = 3
}) => {
  for (let attempt = 0; attempt < maxRetries; attempt++) {
    try {
      return await agent(messages, exampleId);
    } catch (error) {
      // Check for 429, and then if there is a retry-after header
      const {
        isRateLimitError,
        retryAfter
      } = parseErrorMessage(error);
      if (isRateLimitError) {
        logger.error("callAgentWithRetry: Slow down! You're going too fast! 429 detected! Retrying after...");
        if (retryAfter != null) {
          logger.error(`${retryAfter} seconds`);
          await wait(retryAfter * 1000);
          // eslint-disable-next-line no-continue
          continue;
        }
      }
      // If not 429 or there is no retry-after header, reject the promise
      logger.error(`Error calling agent:\n${error}`);
      return Promise.reject(error);
    }
  }
  logger.error(`callAgentWithRetry: Max retries reached: ${maxRetries}`);
  // Reject and keep going!
  // eslint-disable-next-line prefer-promise-reject-errors
  return Promise.reject(`callAgentWithRetry: Max retries reached: ${maxRetries}`);
};
exports.callAgentWithRetry = callAgentWithRetry;
const getMessageFromLangChainResponse = response => {
  if (response.status === 'fulfilled' && response.value.data != null) {
    return getFormattedMessageContent(response.value.data);
  }
  return 'error';
};

/**
 * Lifted from `x-pack/packages/kbn-elastic-assistant/impl/assistant/helpers.ts`
 * TODO: Move this to a shared location
 *
 * When `content` is a JSON string, prefixed with "```json\n"
 * and suffixed with "\n```", this function will attempt to parse it and return
 * the `action_input` property if it exists.
 */
exports.getMessageFromLangChainResponse = getMessageFromLangChainResponse;
const getFormattedMessageContent = content => {
  const formattedContentMatch = content.match(/```json\n([\s\S]+)\n```/);
  if (formattedContentMatch) {
    try {
      var _parsedContent$action;
      const parsedContent = JSON.parse(formattedContentMatch[1]);
      return (_parsedContent$action = parsedContent.action_input) !== null && _parsedContent$action !== void 0 ? _parsedContent$action : content;
    } catch {
      // we don't want to throw an error here, so we'll fall back to the original content
    }
  }
  return content;
};

/**
 * Parse an error message coming back from the agent via the actions frameworks to determine if it is
 * a rate limit error and extract the retry after delay.
 *
 * Note: Could be simplified by instrumenting agents w/ callback where there's access to the actual response
 * @param error
 */
exports.getFormattedMessageContent = getFormattedMessageContent;
const parseErrorMessage = error => {
  const errorMessage = error.message;
  const rateLimitRegex = /Status code: 429.*?Please retry after (\d+) seconds/;
  const match = errorMessage.match(rateLimitRegex);

  // If there is a match, return the parsed delay; otherwise, return an indication that it is not a 429 error.
  if (match && match[1]) {
    return {
      isRateLimitError: true,
      retryAfter: parseInt(match[1], 10)
    };
  } else {
    return {
      isRateLimitError: false,
      retryAfter: null
    };
  }
};
exports.parseErrorMessage = parseErrorMessage;