"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildOtherBucketAgg = exports.OTHER_NESTED_BUCKET_SEPARATOR = void 0;
exports.constructMultiTermOtherFilter = constructMultiTermOtherFilter;
exports.constructSingleTermOtherFilter = constructSingleTermOtherFilter;
exports.updateMissingBucket = exports.mergeOtherBucketAggResponse = exports.createOtherBucketPostFlightRequest = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _rxjs = require("rxjs");
var _agg_groups = require("../agg_groups");
var _sampler = require("../utils/sampler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MISSING_KEY_STRING = '__missing__';
const OTHER_NESTED_BUCKET_SEPARATOR = '╰┄►';
exports.OTHER_NESTED_BUCKET_SEPARATOR = OTHER_NESTED_BUCKET_SEPARATOR;
const otherBucketRegexp = new RegExp(`^${OTHER_NESTED_BUCKET_SEPARATOR}`);

/**
 * walks the aggregation DSL and returns DSL starting at aggregation with id of startFromAggId
 * @param aggNestedDsl: aggregation config DSL (top level)
 * @param startFromId: id of an aggregation from where we want to get the nested DSL
 */
const getNestedAggDSL = (aggNestedDsl, startFromAggId) => {
  if (aggNestedDsl[startFromAggId]) {
    return aggNestedDsl[startFromAggId];
  }
  const nestedAggs = (0, _lodash.values)(aggNestedDsl);
  let aggs;
  for (let i = 0; i < nestedAggs.length; i++) {
    if (nestedAggs[i].aggs && (aggs = getNestedAggDSL(nestedAggs[i].aggs, startFromAggId))) {
      return aggs;
    }
  }
};

/**
 * returns buckets from response for a specific other bucket
 * @param aggConfigs: configuration for the aggregations
 * @param response: response from elasticsearch
 * @param aggWithOtherBucket: AggConfig of the aggregation with other bucket enabled
 * @param key: key from the other bucket request for a specific other bucket
 */
const getAggResultBuckets = (aggConfigs, response, aggWithOtherBucket, key) => {
  var _responseAgg2;
  const keyParts = key.split(OTHER_NESTED_BUCKET_SEPARATOR);
  let responseAgg = response;
  for (const i in keyParts) {
    // enable also the empty string
    if (keyParts[i] != null) {
      const responseAggs = (0, _lodash.values)(responseAgg);
      // If you have multi aggs, we cannot just assume the first one is the `other` bucket,
      // so we need to loop over each agg until we find it.
      for (let aggId = 0; aggId < responseAggs.length; aggId++) {
        const aggById = responseAggs[aggId];
        const aggKey = (0, _lodash.keys)(responseAgg)[aggId];
        const aggConfig = (0, _lodash.find)(aggConfigs.aggs, agg => agg.id === aggKey);
        if (aggConfig) {
          const aggResultBucket = (0, _lodash.find)(aggById.buckets, (bucket, bucketObjKey) => {
            const bucketKey = aggConfig.getKey(bucket, (0, _lodash.isNumber)(bucketObjKey) ? undefined : bucketObjKey).toString();
            return bucketKey === keyParts[i];
          });
          if (aggResultBucket) {
            var _responseAgg;
            // this is a special check in order to avoid an overwrite when
            // there's an empty string term at root level for the data request
            // as the other request will default to empty string category as well
            if (!((_responseAgg = responseAgg) !== null && _responseAgg !== void 0 && _responseAgg[aggWithOtherBucket.id]) || keyParts[i] !== '') {
              responseAgg = aggResultBucket;
              break;
            }
          }
        }
      }
    }
  }
  if ((_responseAgg2 = responseAgg) !== null && _responseAgg2 !== void 0 && _responseAgg2[aggWithOtherBucket.id]) {
    return responseAgg[aggWithOtherBucket.id].buckets;
  }
  return [];
};

/**
 * gets all the missing buckets in our response for a specific aggregation id
 * @param responseAggs: array of aggregations from response
 * @param aggId: id of the aggregation with missing bucket
 */
const getAggConfigResultMissingBuckets = (responseAggs, aggId) => {
  const resultBuckets = [];
  if (responseAggs[aggId]) {
    const matchingBucket = responseAggs[aggId].buckets.find(bucket => bucket.key === MISSING_KEY_STRING);
    if (matchingBucket) {
      resultBuckets.push(matchingBucket);
    }
    return resultBuckets;
  }
  (0, _lodash.each)(responseAggs, agg => {
    if (agg.buckets) {
      (0, _lodash.each)(agg.buckets, bucket => {
        resultBuckets.push(...getAggConfigResultMissingBuckets(bucket, aggId));
      });
    }
  });
  return resultBuckets;
};

/**
 * gets all the terms that are NOT in the other bucket
 * @param requestAgg: an aggregation we are looking at
 * @param key: the key for this specific other bucket
 * @param otherAgg: AggConfig of the aggregation with other bucket
 */
const getOtherAggTerms = (requestAgg, key, otherAgg) => {
  return requestAgg['other-filter'].filters.filters[key].bool.must_not.filter(filter => filter.match_phrase && filter.match_phrase[otherAgg.params.field.name] != null // mind empty strings!
  ).map(filter => filter.match_phrase[otherAgg.params.field.name]);
};

/**
 * Helper function to handle sampling case and get the correct cursor agg from a request object
 */
const getCorrectAggCursorFromRequest = (requestAgg, aggConfigs) => {
  return aggConfigs.isSamplingEnabled() ? requestAgg.sampling.aggs : requestAgg;
};

/**
 * Helper function to handle sampling case and get the correct cursor agg from a response object
 */
const getCorrectAggregationsCursorFromResponse = (response, aggConfigs) => {
  var _response$aggregation;
  return aggConfigs.isSamplingEnabled() ? (_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : _response$aggregation.sampling : response.aggregations;
};
const buildOtherBucketAgg = (aggConfigs, aggWithOtherBucket, response) => {
  const bucketAggs = aggConfigs.aggs.filter(agg => agg.type.type === _agg_groups.AggGroupNames.Buckets && agg.enabled);
  const index = bucketAggs.findIndex(agg => agg.id === aggWithOtherBucket.id);
  const aggs = aggConfigs.toDsl();
  const indexPattern = aggWithOtherBucket.aggConfigs.indexPattern;

  // create filters aggregation
  const filterAgg = aggConfigs.createAggConfig({
    type: 'filters',
    id: 'other',
    params: {
      filters: []
    },
    enabled: false
  }, {
    addToAggConfigs: false
  });

  // nest all the child aggregations of aggWithOtherBucket
  const resultAgg = {
    aggs: getNestedAggDSL(aggs, aggWithOtherBucket.id).aggs,
    filters: filterAgg.toDsl()
  };
  let noAggBucketResults = false;
  let exhaustiveBuckets = true;

  // recursively create filters for all parent aggregation buckets
  const walkBucketTree = (aggIndex, aggregations, aggId, filters, key) => {
    var _aggWithOtherBucket$p;
    // make sure there are actually results for the buckets
    const agg = aggregations[aggId];
    if (!agg || (
    // buckets can be either an array or an object in case there's also a filter at the same level
    Array.isArray(agg.buckets) ? !agg.buckets.length : !Object.values(agg.buckets).length)) {
      noAggBucketResults = true;
      return;
    }
    const newAggIndex = aggIndex + 1;
    const newAgg = bucketAggs[newAggIndex];
    const currentAgg = bucketAggs[aggIndex];
    if (aggIndex === index && agg && agg.sum_other_doc_count > 0) {
      exhaustiveBuckets = false;
    }
    if (aggIndex < index) {
      (0, _lodash.each)(agg === null || agg === void 0 ? void 0 : agg.buckets, (bucket, bucketObjKey) => {
        const bucketKey = currentAgg.getKey(bucket, (0, _lodash.isNumber)(bucketObjKey) ? undefined : bucketObjKey);
        const filter = (0, _lodash.cloneDeep)(bucket.filters) || currentAgg.createFilter(bucketKey);
        const newFilters = (0, _lodash.flatten)([...filters, filter]);
        walkBucketTree(newAggIndex, bucket, newAgg.id, newFilters, `${key}${OTHER_NESTED_BUCKET_SEPARATOR}${bucketKey.toString()}`);
      });
      return;
    }
    const hasScriptedField = !!((_aggWithOtherBucket$p = aggWithOtherBucket.params.field) !== null && _aggWithOtherBucket$p !== void 0 && _aggWithOtherBucket$p.scripted);
    const hasMissingBucket = !!aggWithOtherBucket.params.missingBucket;
    const hasMissingBucketKey = agg.buckets.some(bucket => bucket.key === MISSING_KEY_STRING);
    if (aggWithOtherBucket.params.field && !hasScriptedField && (!hasMissingBucket || hasMissingBucketKey)) {
      filters.push((0, _esQuery.buildExistsFilter)(aggWithOtherBucket.params.field, aggWithOtherBucket.aggConfigs.indexPattern));
    }

    // create not filters for all the buckets
    (0, _lodash.each)(agg.buckets, bucket => {
      if (bucket.key === MISSING_KEY_STRING) return;
      const filter = currentAgg.createFilter(currentAgg.getKey(bucket, bucket.key));
      filter.meta.negate = true;
      filters.push(filter);
    });
    resultAgg.filters.filters[key] = {
      bool: (0, _esQuery.buildQueryFromFilters)(filters, indexPattern)
    };
  };
  walkBucketTree(0, getCorrectAggregationsCursorFromResponse(response, aggConfigs), bucketAggs[0].id, [], '');

  // bail if there were no bucket results
  if (noAggBucketResults || exhaustiveBuckets) {
    return false;
  }
  return () => {
    if (aggConfigs.isSamplingEnabled()) {
      return {
        sampling: {
          ...(0, _sampler.createSamplerAgg)(aggConfigs.samplerConfig),
          aggs: {
            'other-filter': resultAgg
          }
        }
      };
    }
    return {
      'other-filter': resultAgg
    };
  };
};
exports.buildOtherBucketAgg = buildOtherBucketAgg;
const mergeOtherBucketAggResponse = (aggsConfig, response, otherResponse, otherAgg, requestAgg, otherFilterBuilder) => {
  const updatedResponse = (0, _lodash.cloneDeep)(response);
  const aggregationsRoot = getCorrectAggregationsCursorFromResponse(otherResponse, aggsConfig);
  const updatedAggregationsRoot = getCorrectAggregationsCursorFromResponse(updatedResponse, aggsConfig);
  const buckets = 'buckets' in aggregationsRoot['other-filter'] ? aggregationsRoot['other-filter'].buckets : {};
  (0, _lodash.each)(buckets, (bucket, key) => {
    if (!bucket.doc_count || key === undefined) return;
    const bucketKey = key.replace(otherBucketRegexp, '');
    const aggResultBuckets = getAggResultBuckets(aggsConfig, updatedAggregationsRoot, otherAgg, bucketKey);
    const otherFilter = otherFilterBuilder(getCorrectAggCursorFromRequest(requestAgg, aggsConfig), key, otherAgg);
    bucket.filters = [otherFilter];
    bucket.key = '__other__';
    if (aggResultBuckets.some(aggResultBucket => aggResultBucket.key === MISSING_KEY_STRING)) {
      bucket.filters.push((0, _esQuery.buildExistsFilter)(otherAgg.params.field, otherAgg.aggConfigs.indexPattern));
    }
    aggResultBuckets.push(bucket);
  });
  return updatedResponse;
};
exports.mergeOtherBucketAggResponse = mergeOtherBucketAggResponse;
const updateMissingBucket = (response, aggConfigs, agg) => {
  const updatedResponse = (0, _lodash.cloneDeep)(response);
  const aggResultBuckets = getAggConfigResultMissingBuckets(getCorrectAggregationsCursorFromResponse(updatedResponse, aggConfigs), agg.id);
  aggResultBuckets.forEach(bucket => {
    bucket.key = MISSING_KEY_STRING;
  });
  return updatedResponse;
};
exports.updateMissingBucket = updateMissingBucket;
function constructSingleTermOtherFilter(requestAgg, key, otherAgg) {
  const requestFilterTerms = getOtherAggTerms(requestAgg, key, otherAgg);
  const phraseFilter = (0, _esQuery.buildPhrasesFilter)(otherAgg.params.field, requestFilterTerms, otherAgg.aggConfigs.indexPattern);
  phraseFilter.meta.negate = true;
  return phraseFilter;
}
function constructMultiTermOtherFilter(requestAgg, key) {
  return {
    query: requestAgg['other-filter'].filters.filters[key],
    meta: {}
  };
}
const createOtherBucketPostFlightRequest = otherFilterBuilder => {
  const postFlightRequest = async (resp, aggConfigs, aggConfig, searchSource, inspectorRequestAdapter, abortSignal, searchSessionId, disableWarningToasts) => {
    if (!resp.aggregations) return resp;
    const nestedSearchSource = searchSource.createChild();
    if (aggConfig.params.otherBucket) {
      const filterAgg = buildOtherBucketAgg(aggConfigs, aggConfig, resp);
      if (!filterAgg) return resp;
      nestedSearchSource.setField('aggs', filterAgg);
      const {
        rawResponse: otherResponse
      } = await (0, _rxjs.lastValueFrom)(nestedSearchSource.fetch$({
        abortSignal,
        sessionId: searchSessionId,
        disableWarningToasts,
        inspector: {
          adapter: inspectorRequestAdapter,
          title: _i18n.i18n.translate('data.search.aggs.buckets.terms.otherBucketTitle', {
            defaultMessage: 'Other bucket'
          }),
          description: _i18n.i18n.translate('data.search.aggs.buckets.terms.otherBucketDescription', {
            defaultMessage: 'This request counts the number of documents that fall ' + 'outside the criterion of the data buckets.'
          })
        }
      }));
      resp = mergeOtherBucketAggResponse(aggConfigs, resp, otherResponse, aggConfig, filterAgg(), otherFilterBuilder);
    }
    if (aggConfig.params.missingBucket) {
      resp = updateMissingBucket(resp, aggConfigs, aggConfig);
    }
    return resp;
  };
  return postFlightRequest;
};
exports.createOtherBucketPostFlightRequest = createOtherBucketPostFlightRequest;