"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldListMapFromMappingDefinitions = exports.getFieldListFromTypeMapping = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Return the list of fields present in each individual type mappings present in the definition.
 */
const getFieldListMapFromMappingDefinitions = mappings => {
  return Object.entries(mappings).reduce((memo, [typeName, typeMappings]) => {
    memo[typeName] = getFieldListFromTypeMapping(typeMappings);
    return memo;
  }, {});
};
exports.getFieldListMapFromMappingDefinitions = getFieldListMapFromMappingDefinitions;
/**
 * Return the list of fields present in the provided mappings.
 * Note that fields only containing properties are still considered fields by this function.
 *
 * @example
 * ```
 * getFieldListFromTypeMapping({
 *   properties: {
 *     foo: {
 *       properties: {
 *         hello: { type: 'text' },
 *         dolly: { type: 'text' },
 *       },
 *     },
 *   },
 * });
 * // ['foo', 'foo.dolly', 'foo.hello']
 * ```
 */
const getFieldListFromTypeMapping = typeMappings => {
  const fieldList = [];
  const queue = [];
  Object.entries(typeMappings.properties).forEach(([fieldName, fieldDef]) => {
    queue.push({
      fieldPath: [fieldName],
      fieldDef
    });
  });
  while (queue.length > 0) {
    const item = queue.pop();
    fieldList.push(item.fieldPath.join('.'));
    if ('properties' in item.fieldDef) {
      var _item$fieldDef$proper;
      Object.entries((_item$fieldDef$proper = item.fieldDef.properties) !== null && _item$fieldDef$proper !== void 0 ? _item$fieldDef$proper : {}).forEach(([fieldName, fieldDef]) => {
        queue.push({
          fieldPath: [...item.fieldPath, fieldName],
          fieldDef
        });
      });
    }
  }
  return fieldList.sort();
};
exports.getFieldListFromTypeMapping = getFieldListFromTypeMapping;