"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildPalettes = void 0;
var _chromaJs = _interopRequireDefault(require("chroma-js"));
var _i18n = require("@kbn/i18n");
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _ = require("../..");
var _lighten_color = require("./lighten_color");
var _mapped_colors = require("../mapped_colors");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function buildRoundRobinCategoricalWithMappedColors() {
  const colors = (0, _eui.euiPaletteColorBlind)({
    rotations: 2
  });
  const behindTextColors = (0, _eui.euiPaletteColorBlindBehindText)({
    rotations: 2
  });
  const behindTextColorMap = Object.fromEntries((0, _lodash.zip)(colors, behindTextColors));
  const mappedColors = new _mapped_colors.MappedColors(undefined, num => {
    return (0, _lodash.flatten)(new Array(Math.ceil(num / 10)).fill(colors)).map(color => color.toLowerCase());
  });
  function getColor(series, chartConfiguration = {
    behindText: false
  }) {
    let outputColor;
    if (chartConfiguration.syncColors) {
      const colorKey = series[0].name;
      mappedColors.mapKeys([colorKey]);
      const mappedColor = mappedColors.get(colorKey);
      outputColor = chartConfiguration.behindText ? behindTextColorMap[mappedColor] : mappedColor;
    } else {
      outputColor = chartConfiguration.behindText ? behindTextColors[series[0].rankAtDepth % behindTextColors.length] : colors[series[0].rankAtDepth % colors.length];
    }
    if (!chartConfiguration.maxDepth || chartConfiguration.maxDepth === 1) {
      return outputColor;
    }
    return (0, _lighten_color.lightenColor)(outputColor, series.length, chartConfiguration.maxDepth);
  }
  return {
    id: 'default',
    getCategoricalColor: getColor,
    getCategoricalColors: () => (0, _eui.euiPaletteColorBlind)(),
    toExpression: () => ({
      type: 'expression',
      chain: [{
        type: 'function',
        function: 'system_palette',
        arguments: {
          name: ['default']
        }
      }]
    })
  };
}
function buildGradient(id, colors) {
  function getColor(series, chartConfiguration = {
    behindText: false
  }) {
    const totalSeriesAtDepth = series[0].totalSeriesAtDepth;
    const rankAtDepth = series[0].rankAtDepth;
    const actualColors = colors(totalSeriesAtDepth);
    const outputColor = actualColors[rankAtDepth];
    if (!chartConfiguration.maxDepth || chartConfiguration.maxDepth === 1) {
      return outputColor;
    }
    return (0, _lighten_color.lightenColor)(outputColor, series.length, chartConfiguration.maxDepth);
  }
  return {
    id,
    getCategoricalColor: getColor,
    getCategoricalColors: colors,
    canDynamicColoring: true,
    toExpression: () => ({
      type: 'expression',
      chain: [{
        type: 'function',
        function: 'system_palette',
        arguments: {
          name: [id]
        }
      }]
    })
  };
}
function buildSyncedKibanaPalette(colors) {
  const staticColors = (0, _.createColorPalette)(20);
  function getColor(series, chartConfiguration = {}) {
    let outputColor;
    if (chartConfiguration.syncColors) {
      colors.mappedColors.mapKeys([series[0].name]);
      outputColor = colors.mappedColors.get(series[0].name);
    } else {
      const configColor = colors.mappedColors.getColorFromConfig(series[0].name);
      outputColor = configColor || staticColors[series[0].rankAtDepth % staticColors.length];
    }
    if (!chartConfiguration.maxDepth || chartConfiguration.maxDepth === 1) {
      return outputColor;
    }
    return (0, _lighten_color.lightenColor)(outputColor, series.length, chartConfiguration.maxDepth);
  }
  return {
    id: 'kibana_palette',
    getCategoricalColor: getColor,
    getCategoricalColors: () => colors.seedColors.slice(0, 10),
    toExpression: () => ({
      type: 'expression',
      chain: [{
        type: 'function',
        function: 'system_palette',
        arguments: {
          name: ['kibana_palette']
        }
      }]
    })
  };
}
function buildCustomPalette() {
  return {
    id: 'custom',
    getColorForValue: (value, params, dataBounds) => {
      return (0, _helpers.workoutColorForValue)(value, params, dataBounds);
    },
    getCategoricalColor: (series, chartConfiguration = {
      behindText: false
    }, {
      colors,
      gradient
    }) => {
      const actualColors = gradient ? _chromaJs.default.scale(colors).colors(series[0].totalSeriesAtDepth) : colors;
      const outputColor = actualColors[series[0].rankAtDepth % actualColors.length];
      if (!chartConfiguration.maxDepth || chartConfiguration.maxDepth === 1) {
        return outputColor;
      }
      return (0, _lighten_color.lightenColor)(outputColor, series.length, chartConfiguration.maxDepth);
    },
    internal: true,
    title: _i18n.i18n.translate('charts.palettes.customLabel', {
      defaultMessage: 'Custom'
    }),
    getCategoricalColors: (size, {
      colors,
      gradient,
      stepped,
      stops
    } = {
      colors: [],
      gradient: false,
      stepped: false,
      stops: []
    }) => {
      if (stepped) {
        const range = stops[stops.length - 1] - stops[0];
        const offset = stops[0];
        const finalStops = [...stops.map(stop => (stop - offset) / range)];
        return _chromaJs.default.scale(colors).domain(finalStops).colors(size);
      }
      return gradient ? _chromaJs.default.scale(colors).colors(size) : colors;
    },
    canDynamicColoring: false,
    toExpression: ({
      colors,
      gradient,
      stops = [],
      rangeMax,
      rangeMin,
      rangeType = 'percent',
      continuity = 'above',
      reverse = false
    }) => ({
      type: 'expression',
      chain: [{
        type: 'function',
        function: 'palette',
        arguments: {
          color: colors,
          gradient: [gradient],
          reverse: [reverse],
          continuity: [continuity],
          stop: stops,
          range: [rangeType],
          rangeMax: rangeMax == null ? [] : [rangeMax],
          rangeMin: rangeMin == null ? [] : [rangeMin]
        }
      }]
    })
  };
}
const buildPalettes = legacyColorsService => {
  return {
    default: {
      title: _i18n.i18n.translate('charts.palettes.defaultPaletteLabel', {
        defaultMessage: 'Default'
      }),
      ...buildRoundRobinCategoricalWithMappedColors()
    },
    status: {
      title: _i18n.i18n.translate('charts.palettes.statusLabel', {
        defaultMessage: 'Status'
      }),
      ...buildGradient('status', _eui.euiPaletteForStatus)
    },
    temperature: {
      title: _i18n.i18n.translate('charts.palettes.temperatureLabel', {
        defaultMessage: 'Temperature'
      }),
      ...buildGradient('temperature', _eui.euiPaletteForTemperature)
    },
    complimentary: {
      title: _i18n.i18n.translate('charts.palettes.complimentaryLabel', {
        defaultMessage: 'Complimentary'
      }),
      ...buildGradient('complimentary', _eui.euiPaletteComplementary)
    },
    negative: {
      title: _i18n.i18n.translate('charts.palettes.negativeLabel', {
        defaultMessage: 'Negative'
      }),
      ...buildGradient('negative', _eui.euiPaletteNegative)
    },
    positive: {
      title: _i18n.i18n.translate('charts.palettes.positiveLabel', {
        defaultMessage: 'Positive'
      }),
      ...buildGradient('positive', _eui.euiPalettePositive)
    },
    cool: {
      title: _i18n.i18n.translate('charts.palettes.coolLabel', {
        defaultMessage: 'Cool'
      }),
      ...buildGradient('cool', _eui.euiPaletteCool)
    },
    warm: {
      title: _i18n.i18n.translate('charts.palettes.warmLabel', {
        defaultMessage: 'Warm'
      }),
      ...buildGradient('warm', _eui.euiPaletteWarm)
    },
    gray: {
      title: _i18n.i18n.translate('charts.palettes.grayLabel', {
        defaultMessage: 'Gray'
      }),
      ...buildGradient('gray', _eui.euiPaletteGray)
    },
    kibana_palette: {
      title: _i18n.i18n.translate('charts.palettes.kibanaPaletteLabel', {
        defaultMessage: 'Compatibility'
      }),
      ...buildSyncedKibanaPalette(legacyColorsService)
    },
    custom: buildCustomPalette()
  };
};
exports.buildPalettes = buildPalettes;