"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCreateCaseWithAttachmentsTransaction = exports.useAddAttachmentToExistingCaseTransaction = void 0;
var _react = require("react");
var _domain = require("../../../common/types/domain");
var _use_start_transaction = require("./use_start_transaction");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CREATE_CASE = 'createCase';
const ADD_ATTACHMENT_TO_NEW_CASE = 'addAttachmentToNewCase';
const BULK_ADD_ATTACHMENT_TO_NEW_CASE = 'bulkAddAttachmentsToNewCase';
const ADD_ATTACHMENT_TO_EXISTING_CASE = 'addAttachmentToExistingCase';
const BULK_ADD_ATTACHMENT_TO_EXISTING_CASE = 'bulkAddAttachmentsToExistingCase';
// Called when a case is created, attachments are optional
const useCreateCaseWithAttachmentsTransaction = () => {
  const {
    startTransaction
  } = (0, _use_start_transaction.useStartTransaction)();
  const startCreateCaseWithAttachmentsTransaction = (0, _react.useCallback)(({
    appId,
    attachments
  }) => {
    if (!attachments) {
      return startTransaction(`Cases [${appId}] ${CREATE_CASE}`);
    }
    const alertCount = getAlertCount(attachments);
    if (alertCount <= 1) {
      return startTransaction(`Cases [${appId}] ${ADD_ATTACHMENT_TO_NEW_CASE}`);
    }
    const transaction = startTransaction(`Cases [${appId}] ${BULK_ADD_ATTACHMENT_TO_NEW_CASE}`);
    transaction === null || transaction === void 0 ? void 0 : transaction.addLabels({
      alert_count: alertCount
    });
    return transaction;
  }, [startTransaction]);
  return {
    startTransaction: startCreateCaseWithAttachmentsTransaction
  };
};
exports.useCreateCaseWithAttachmentsTransaction = useCreateCaseWithAttachmentsTransaction;
// Called when attachments are added to existing case
const useAddAttachmentToExistingCaseTransaction = () => {
  const {
    startTransaction
  } = (0, _use_start_transaction.useStartTransaction)();
  const startAddAttachmentToExistingCaseTransaction = (0, _react.useCallback)(({
    appId,
    attachments
  }) => {
    const alertCount = getAlertCount(attachments);
    if (alertCount <= 1) {
      return startTransaction(`Cases [${appId}] ${ADD_ATTACHMENT_TO_EXISTING_CASE}`);
    }
    const transaction = startTransaction(`Cases [${appId}] ${BULK_ADD_ATTACHMENT_TO_EXISTING_CASE}`);
    transaction === null || transaction === void 0 ? void 0 : transaction.addLabels({
      alert_count: alertCount
    });
    return transaction;
  }, [startTransaction]);
  return {
    startTransaction: startAddAttachmentToExistingCaseTransaction
  };
};
exports.useAddAttachmentToExistingCaseTransaction = useAddAttachmentToExistingCaseTransaction;
const getAlertCount = attachments => {
  return attachments.reduce((total, attachment) => {
    if (attachment.type !== _domain.AttachmentType.alert) {
      return total;
    }
    if (!Array.isArray(attachment.alertId)) {
      return total + 1;
    }
    return total + attachment.alertId.length;
  }, 0);
};