"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectServices = collectServices;
var _debug_log = require("../../../common/debug_log");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function collectServices({
  client,
  from,
  to,
  sourceIndices,
  afterKey,
  filters = []
}) {
  var _esResponse$aggregati;
  if (!(sourceIndices !== null && sourceIndices !== void 0 && sourceIndices.apm)) {
    throw new Error('missing required apm indices');
  }
  const {
    transaction,
    error,
    metric
  } = sourceIndices.apm;
  const musts = [...filters, {
    exists: {
      field: 'service.name'
    }
  }];
  const dsl = {
    index: [transaction, error, metric],
    size: 0,
    _source: false,
    query: {
      bool: {
        filter: [{
          range: {
            '@timestamp': {
              gte: from,
              lte: to
            }
          }
        }],
        must: musts
      }
    },
    aggs: {
      services: {
        composite: {
          size: _.QUERY_MAX_SIZE,
          sources: [{
            serviceName: {
              terms: {
                field: 'service.name'
              }
            }
          }, {
            serviceEnvironment: {
              terms: {
                field: 'service.environment',
                missing_bucket: true
              }
            }
          }]
        },
        aggs: {
          container_and_hosts: {
            multi_terms: {
              terms: [{
                field: 'host.hostname'
              }, {
                field: 'container.id'
              }]
            }
          }
        }
      }
    }
  };
  if (afterKey) {
    dsl.aggs.services.composite.after = afterKey;
  }
  (0, _debug_log.debug)(dsl);
  const esResponse = await client.search(dsl);
  const {
    after_key: nextKey,
    buckets = []
  } = ((_esResponse$aggregati = esResponse.aggregations) === null || _esResponse$aggregati === void 0 ? void 0 : _esResponse$aggregati.services) || {};
  const assets = buckets.reduce((acc, bucket) => {
    var _containerHosts$bucke;
    const {
      key: {
        serviceName,
        serviceEnvironment
      },
      container_and_hosts: containerHosts
    } = bucket;
    if (!serviceName) {
      return acc;
    }
    const service = {
      '@timestamp': new Date().toISOString(),
      'asset.kind': 'service',
      'asset.id': serviceName,
      'asset.ean': `service:${serviceName}`,
      'asset.references': [],
      'asset.parents': []
    };
    if (serviceEnvironment) {
      service['service.environment'] = serviceEnvironment;
    }
    (_containerHosts$bucke = containerHosts.buckets) === null || _containerHosts$bucke === void 0 ? void 0 : _containerHosts$bucke.forEach(containerBucket => {
      const [hostname, containerId] = containerBucket.key;
      if (hostname) {
        service['asset.references'].push(`host:${hostname}`);
      }
      if (containerId) {
        service['asset.parents'].push(`container:${containerId}`);
      }
    });
    acc.push(service);
    return acc;
  }, []);
  return {
    assets,
    afterKey: buckets.length === _.QUERY_MAX_SIZE ? nextKey : undefined
  };
}