"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addApiKeysToEveryPackagePolicyIfMissing = addApiKeysToEveryPackagePolicyIfMissing;
exports.addApiKeysToPackagePolicyIfMissing = addApiKeysToPackagePolicyIfMissing;
var _lodash = require("lodash");
var _get_internal_saved_objects_client = require("../../../lib/helpers/get_internal_saved_objects_client");
var _get_apm_package_policies = require("../get_apm_package_policies");
var _create_apm_api_keys = require("./create_apm_api_keys");
var _get_package_policy_decorators = require("../get_package_policy_decorators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function addApiKeysToEveryPackagePolicyIfMissing({
  coreStartPromise,
  pluginStartPromise,
  logger
}) {
  const coreStart = await coreStartPromise;
  const {
    fleet
  } = await pluginStartPromise;
  if (!fleet) {
    return;
  }
  const apmFleetPolicies = await (0, _get_apm_package_policies.getApmPackagePolicies)({
    coreStart,
    fleetPluginStart: fleet
  });
  return Promise.all(apmFleetPolicies.items.map(policy => {
    return addApiKeysToPackagePolicyIfMissing({
      policy,
      coreStart,
      fleet,
      logger
    });
  }));
}
async function addApiKeysToPackagePolicyIfMissing({
  policy,
  coreStart,
  fleet,
  logger
}) {
  if ((0, _get_package_policy_decorators.policyHasApiKey)(policy)) {
    logger.debug(`Policy (${policy.id}) already has api key`);
    return;
  }
  logger.debug(`Policy (${policy.id}) does not have api key`);
  const agentConfigApiKey = await (0, _create_apm_api_keys.createApmAgentConfigApiKey)({
    coreStart,
    logger,
    packagePolicyId: policy.id
  });
  const sourceMapApiKey = await (0, _create_apm_api_keys.createApmSourceMapApiKey)({
    coreStart,
    logger,
    packagePolicyId: policy.id
  });
  const packagePolicyTrimmed = (0, _lodash.omit)(policy, ['id', 'revision', 'updated_at', 'updated_by']);
  const policyWithApiKeys = (0, _get_package_policy_decorators.getPackagePolicyWithApiKeys)({
    packagePolicy: packagePolicyTrimmed,
    agentConfigApiKey,
    sourceMapApiKey
  });
  const internalESClient = coreStart.elasticsearch.client.asInternalUser;
  const savedObjectsClient = await (0, _get_internal_saved_objects_client.getInternalSavedObjectsClient)(coreStart);
  const newPolicy = await fleet.packagePolicyService.update(savedObjectsClient, internalESClient, policy.id, policyWithApiKeys);
  logger.debug(`Added api keys to policy ${policy.id}`);
  return newPolicy;
}