"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSignificantItemGroups = getSignificantItemGroups;
var _duplicate_identifier = require("./duplicate_identifier");
var _fetch_frequent_item_sets = require("./fetch_frequent_item_sets");
var _get_field_value_pair_counts = require("./get_field_value_pair_counts");
var _get_marked_duplicates = require("./get_marked_duplicates");
var _get_simple_hierarchical_tree = require("./get_simple_hierarchical_tree");
var _get_simple_hierarchical_tree_leaves = require("./get_simple_hierarchical_tree_leaves");
var _get_missing_significant_items = require("./get_missing_significant_items");
var _transform_significant_item_to_group = require("./transform_significant_item_to_group");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getSignificantItemGroups(itemsets, significantItems, fields) {
  // We use the grouped significant items to later repopulate
  // the `frequent_item_sets` result with the missing duplicates.
  const groupedSignificantItems = (0, _fetch_frequent_item_sets.groupDuplicates)(significantItems, _duplicate_identifier.duplicateIdentifier).filter(g => g.group.length > 1);

  // `frequent_item_sets` returns lot of different small groups of field/value pairs that co-occur.
  // The following steps analyse these small groups, identify overlap between these groups,
  // and then summarize them in larger groups where possible.

  // Get a tree structure based on `frequent_item_sets`.
  const {
    root
  } = (0, _get_simple_hierarchical_tree.getSimpleHierarchicalTree)(itemsets, false, false, significantItems, fields);

  // Each leave of the tree will be a summarized group of co-occuring field/value pairs.
  const treeLeaves = (0, _get_simple_hierarchical_tree_leaves.getSimpleHierarchicalTreeLeaves)(root, []);

  // To be able to display a more cleaned up results table in the UI, we identify field/value pairs
  // that occur in multiple groups. This will allow us to highlight field/value pairs that are
  // unique to a group in a better way.
  const fieldValuePairCounts = (0, _get_field_value_pair_counts.getFieldValuePairCounts)(treeLeaves);
  const significantItemGroups = (0, _get_marked_duplicates.getMarkedDuplicates)(treeLeaves, fieldValuePairCounts);

  // Some field/value pairs might not be part of the `frequent_item_sets` result set, for example
  // because they don't co-occur with other field/value pairs or because of the limits we set on the query.
  // In this next part we identify those missing pairs and add them as individual groups.
  const missingSignificantItems = (0, _get_missing_significant_items.getMissingSignificantItems)(significantItems, significantItemGroups);
  significantItemGroups.push(...missingSignificantItems.map(significantItem => (0, _transform_significant_item_to_group.transformSignificantItemToGroup)(significantItem, groupedSignificantItems)));
  return significantItemGroups;
}