"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupingHandlerFactory = void 0;
var _async = require("async");
var _fieldTypes = require("@kbn/field-types");
var _i18n = require("@kbn/i18n");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _constants = require("../../../../common/constants");
var _actions = require("../../../../common/api/log_rate_analysis/actions");
var _is_request_aborted_error = require("../../../lib/is_request_aborted_error");
var _fetch_frequent_item_sets = require("../queries/fetch_frequent_item_sets");
var _fetch_terms_2_categories_counts = require("../queries/fetch_terms_2_categories_counts");
var _get_group_filter = require("../queries/get_group_filter");
var _get_histogram_query = require("../queries/get_histogram_query");
var _get_significant_item_groups = require("../queries/get_significant_item_groups");
var _constants2 = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const groupingHandlerFactory = ({
  abortSignal,
  client,
  requestBody,
  responseStream,
  logDebugMessage,
  logger,
  stateHandler,
  version
}) => async (significantCategories, significantTerms, overallTimeSeries) => {
  logDebugMessage('Group results.');
  function pushHistogramDataLoadingState() {
    responseStream.push((0, _actions.updateLoadingStateAction)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.loadingHistogramData', {
        defaultMessage: 'Loading histogram data.'
      })
    }));
  }
  responseStream.push((0, _actions.updateLoadingStateAction)({
    ccsWarning: false,
    loaded: stateHandler.loaded(),
    loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.groupingResults', {
      defaultMessage: 'Transforming significant field/value pairs into groups.'
    }),
    groupsMissing: true
  }));
  try {
    const {
      fields,
      itemSets
    } = await (0, _fetch_frequent_item_sets.fetchFrequentItemSets)(client, requestBody.index, JSON.parse(requestBody.searchQuery), significantTerms, requestBody.timeFieldName, requestBody.deviationMin, requestBody.deviationMax, logger, stateHandler.sampleProbability(), responseStream.pushError, abortSignal);
    if (significantCategories.length > 0 && significantTerms.length > 0) {
      const {
        fields: significantCategoriesFields,
        itemSets: significantCategoriesItemSets
      } = await (0, _fetch_terms_2_categories_counts.fetchTerms2CategoriesCounts)(client, requestBody, JSON.parse(requestBody.searchQuery), significantTerms, itemSets, significantCategories, requestBody.deviationMin, requestBody.deviationMax, logger, responseStream.pushError, abortSignal);
      fields.push(...significantCategoriesFields);
      itemSets.push(...significantCategoriesItemSets);
    }
    if (stateHandler.shouldStop()) {
      logDebugMessage('shouldStop after fetching frequent_item_sets.');
      responseStream.end();
      return;
    }
    if (fields.length > 0 && itemSets.length > 0) {
      const significantItemGroups = (0, _get_significant_item_groups.getSignificantItemGroups)(itemSets, [...significantTerms, ...significantCategories], fields);

      // We'll find out if there's at least one group with at least two items,
      // only then will we return the groups to the clients and make the grouping option available.
      const maxItems = Math.max(...significantItemGroups.map(g => g.group.length));
      if (maxItems > 1) {
        responseStream.push((0, _actions.addSignificantItemsGroupAction)(significantItemGroups, version));
      }
      stateHandler.loaded(_constants2.PROGRESS_STEP_GROUPING, false);
      pushHistogramDataLoadingState();
      if (stateHandler.shouldStop()) {
        logDebugMessage('shouldStop after grouping.');
        responseStream.end();
        return;
      }
      logDebugMessage(`Fetch ${significantItemGroups.length} group histograms.`);
      const groupHistogramQueue = (0, _async.queue)(async function (cpg) {
        if (stateHandler.shouldStop()) {
          logDebugMessage('shouldStop abort fetching group histograms.');
          groupHistogramQueue.kill();
          responseStream.end();
          return;
        }
        if (overallTimeSeries !== undefined) {
          var _overallTimeSeries$da;
          const histogramQuery = (0, _get_histogram_query.getHistogramQuery)(requestBody, (0, _get_group_filter.getGroupFilter)(cpg));
          let cpgTimeSeries;
          try {
            cpgTimeSeries = (await (0, _mlAggUtils.fetchHistogramsForFields)(client, requestBody.index, histogramQuery,
            // fields
            [{
              fieldName: requestBody.timeFieldName,
              type: _fieldTypes.KBN_FIELD_TYPES.DATE,
              interval: overallTimeSeries.interval,
              min: overallTimeSeries.stats[0],
              max: overallTimeSeries.stats[1]
            }],
            // samplerShardSize
            -1, undefined, abortSignal, stateHandler.sampleProbability(), _constants.RANDOM_SAMPLER_SEED))[0];
          } catch (e) {
            if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
              logger.error(`Failed to fetch the histogram data for group #${cpg.id}, got: \n${e.toString()}`);
              responseStream.pushError(`Failed to fetch the histogram data for group #${cpg.id}.`);
            }
            return;
          }
          const histogram = (_overallTimeSeries$da = overallTimeSeries.data.map(o => {
            var _cpgTimeSeries$data$f, _o$key_as_string2;
            const current = (_cpgTimeSeries$data$f = cpgTimeSeries.data.find(d1 => d1.key_as_string === o.key_as_string)) !== null && _cpgTimeSeries$data$f !== void 0 ? _cpgTimeSeries$data$f : {
              doc_count: 0
            };
            if (version === '1') {
              var _o$key_as_string;
              return {
                key: o.key,
                key_as_string: (_o$key_as_string = o.key_as_string) !== null && _o$key_as_string !== void 0 ? _o$key_as_string : '',
                doc_count_significant_term: current.doc_count,
                doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
              };
            }
            return {
              key: o.key,
              key_as_string: (_o$key_as_string2 = o.key_as_string) !== null && _o$key_as_string2 !== void 0 ? _o$key_as_string2 : '',
              doc_count_significant_item: current.doc_count,
              doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
            };
          })) !== null && _overallTimeSeries$da !== void 0 ? _overallTimeSeries$da : [];
          responseStream.push((0, _actions.addSignificantItemsGroupHistogramAction)([{
            id: cpg.id,
            histogram
          }], version));
        }
      }, _constants2.MAX_CONCURRENT_QUERIES);
      groupHistogramQueue.push(significantItemGroups);
      await groupHistogramQueue.drain();
    }
  } catch (e) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
      logger.error(`Failed to transform field/value pairs into groups, got: \n${e.toString()}`);
      responseStream.pushError(`Failed to transform field/value pairs into groups.`);
    }
  }
};
exports.groupingHandlerFactory = groupingHandlerFactory;