"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloseVectorWeb = void 0;
const closevector_web_1 = require("closevector-web");
const common_js_1 = require("./common.cjs");
const document_js_1 = require("../../document.cjs");
/**
 * Class that implements a vector store using CloseVector, It extends the SaveableVectorStore class and
 * provides methods for adding documents and vectors, performing
 * similarity searches, and saving and loading the vector store.
 */
class CloseVectorWeb extends common_js_1.CloseVector {
    constructor(embeddings, args, credentials) {
        super(embeddings, args, credentials);
        if (args.instance) {
            this.instance = args.instance;
        }
        else {
            this.instance = new closevector_web_1.CloseVectorHNSWWeb(embeddings, args);
        }
    }
    /**
     * Method to save the index to the CloseVector CDN.
     * @param options
     * @param options.url the upload url generated by the CloseVector API: https://closevector-docs.getmegaportal.com/docs/api/http-api/file-url
     * @param options.onProgress a callback function to track the upload progress
     */
    async saveToCloud(options) {
        if (!this.instance.uuid && !options.uuid) {
            throw new Error("No uuid provided");
        }
        if (!this.instance.uuid) {
            this.instance._uuid = options.uuid;
        }
        await this.save(this.instance.uuid);
        await this.instance.saveToCloud(options);
    }
    /**
     * Method to load the index from the CloseVector CDN.
     * @param options
     * @param options.url the upload url generated by the CloseVector API: https://closevector-docs.getmegaportal.com/docs/api/http-api/file-url
     * @param options.onProgress a callback function to track the upload progress
     * @param options.uuid the uuid of the index to be downloaded
     * @param options.embeddings the embeddings to be used by the CloseVectorWeb instance
     */
    static async loadFromCloud(options) {
        const instance = await closevector_web_1.CloseVectorHNSWWeb.loadFromCloud(options);
        const vectorstore = new this(options.embeddings, instance.args, options.credentials);
        return vectorstore;
    }
    /**
     * Static method to load a vector store from a directory. It reads the
     * HNSW index, the arguments, and the document store from the directory,
     * then creates a new CloseVectorWeb instance with these values.
     * @param directory The directory from which to load the vector store.
     * @param embeddings The embeddings to be used by the CloseVectorWeb instance.
     * @returns A Promise that resolves to a new CloseVectorWeb instance.
     */
    static async load(directory, embeddings, credentials) {
        const instance = await closevector_web_1.CloseVectorHNSWWeb.load(directory, embeddings);
        const vectorstore = new this(embeddings, instance.args, credentials);
        return vectorstore;
    }
    /**
     * Static method to create a new CloseVectorWeb instance from texts and metadata.
     * It creates a new Document instance for each text and metadata, then
     * calls the fromDocuments method to create the CloseVectorWeb instance.
     * @param texts The texts to be used to create the documents.
     * @param metadatas The metadata to be used to create the documents.
     * @param embeddings The embeddings to be used by the CloseVectorWeb instance.
     * @param args An optional configuration object for the CloseVectorWeb instance.
     * @param credential An optional credential object for the CloseVector API.
     * @returns A Promise that resolves to a new CloseVectorWeb instance.
     */
    static async fromTexts(texts, metadatas, embeddings, args, credential) {
        const docs = [];
        for (let i = 0; i < texts.length; i += 1) {
            const metadata = Array.isArray(metadatas) ? metadatas[i] : metadatas;
            const newDoc = new document_js_1.Document({
                pageContent: texts[i],
                metadata,
            });
            docs.push(newDoc);
        }
        return await CloseVectorWeb.fromDocuments(docs, embeddings, args, credential);
    }
    /**
     * Static method to create a new CloseVectorWeb instance from documents. It
     * creates a new CloseVectorWeb instance, adds the documents to it, then returns
     * the instance.
     * @param docs The documents to be added to the CloseVectorWeb instance.
     * @param embeddings The embeddings to be used by the CloseVectorWeb instance.
     * @param args An optional configuration object for the CloseVectorWeb instance.
     * @param credentials An optional credential object for the CloseVector API.
     * @returns A Promise that resolves to a new CloseVectorWeb instance.
     */
    static async fromDocuments(docs, embeddings, args, credentials) {
        const _args = args || {
            space: "cosine",
        };
        const instance = new this(embeddings, _args, credentials);
        await instance.addDocuments(docs);
        return instance;
    }
    static async imports() {
        return closevector_web_1.CloseVectorHNSWWeb.imports();
    }
}
exports.CloseVectorWeb = CloseVectorWeb;
