/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { __awaiter, __generator } from "tslib";
import { millisToHrTime } from '@opentelemetry/core';
import { FlatMap } from '../utils';
/**
 * An internal opaque interface that the MetricReader receives as
 * MetricProducer. It acts as the storage key to the internal metric stream
 * state for each MetricReader.
 */
var MetricCollector = /** @class */ (function () {
    function MetricCollector(_sharedState, _metricReader) {
        this._sharedState = _sharedState;
        this._metricReader = _metricReader;
    }
    MetricCollector.prototype.collect = function (options) {
        return __awaiter(this, void 0, void 0, function () {
            var collectionTime, meterCollectionPromises, result;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        collectionTime = millisToHrTime(Date.now());
                        meterCollectionPromises = Array.from(this._sharedState.meterSharedStates.values()).map(function (meterSharedState) {
                            return meterSharedState.collect(_this, collectionTime, options);
                        });
                        return [4 /*yield*/, Promise.all(meterCollectionPromises)];
                    case 1:
                        result = _a.sent();
                        return [2 /*return*/, {
                                resourceMetrics: {
                                    resource: this._sharedState.resource,
                                    scopeMetrics: result.map(function (it) { return it.scopeMetrics; }),
                                },
                                errors: FlatMap(result, function (it) { return it.errors; }),
                            }];
                }
            });
        });
    };
    /**
     * Delegates for MetricReader.forceFlush.
     */
    MetricCollector.prototype.forceFlush = function (options) {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this._metricReader.forceFlush(options)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Delegates for MetricReader.shutdown.
     */
    MetricCollector.prototype.shutdown = function (options) {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this._metricReader.shutdown(options)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    MetricCollector.prototype.selectAggregationTemporality = function (instrumentType) {
        return this._metricReader.selectAggregationTemporality(instrumentType);
    };
    MetricCollector.prototype.selectAggregation = function (instrumentType) {
        return this._metricReader.selectAggregation(instrumentType);
    };
    return MetricCollector;
}());
export { MetricCollector };
//# sourceMappingURL=MetricCollector.js.map