"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToLens = void 0;
var _uuid = require("uuid");
var _common = require("@kbn/data-plugin/common");
var _convert_to_lens = require("@kbn/visualizations-plugin/common/convert_to_lens");
var _enums = require("../../../common/enums");
var _services = require("../../services");
var _table = require("../lib/configurations/table");
var _datasource = require("../lib/datasource");
var _series = require("../lib/series");
var _metrics = require("../lib/metrics");
var _convert = require("../lib/convert");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const excludeMetaFromLayers = layers => {
  const newLayers = {};
  Object.entries(layers).forEach(([layerId, layer]) => {
    const columns = layer.columns.map(_convert.excludeMetaFromColumn);
    newLayers[layerId] = {
      ...layer,
      columns
    };
  });
  return newLayers;
};
const invalidModelError = () => new Error('Invalid model');
const convertToLens = async ({
  params: model,
  uiState
}, timeRange) => {
  const columnStates = [];
  const dataViews = (0, _services.getDataViewsStart)();
  try {
    var _uiState$get$sort, _uiState$get;
    const visibleSeries = model.series.filter(series => !series.hidden);
    const seriesNum = visibleSeries.length;
    const sortConfig = (_uiState$get$sort = (_uiState$get = uiState.get('table')) === null || _uiState$get === void 0 ? void 0 : _uiState$get.sort) !== null && _uiState$get$sort !== void 0 ? _uiState$get$sort : {};
    const datasourceInfo = await (0, _datasource.extractOrGenerateDatasourceInfo)(model.index_pattern, model.time_field, false, undefined, undefined, dataViews);
    if (!datasourceInfo) {
      throw invalidModelError();
    }
    const {
      indexPatternId,
      indexPattern
    } = datasourceInfo;
    const commonBucketsColumns = (0, _series.getBucketsColumns)(undefined, {
      split_mode: 'terms',
      terms_field: model.pivot_id,
      terms_size: model.pivot_rows ? model.pivot_rows.toString() : undefined
    }, [], indexPattern, false, model.pivot_label, false);
    if (!commonBucketsColumns) {
      throw invalidModelError();
    }
    const sortConfiguration = {
      columnId: commonBucketsColumns[0].columnId,
      direction: sortConfig.order
    };
    columnStates.push((0, _table.getColumnState)(commonBucketsColumns[0].columnId));
    let bucketsColumns = [];
    if (!model.series.every(s => (!s.aggregate_by && !model.series[0].aggregate_by || s.aggregate_by === model.series[0].aggregate_by) && (!s.aggregate_function && !model.series[0].aggregate_function || s.aggregate_function === model.series[0].aggregate_function))) {
      throw invalidModelError();
    }
    if (model.series[0].aggregate_by) {
      if (!model.series[0].aggregate_function || !['sum', 'mean', 'min', 'max'].includes(model.series[0].aggregate_function)) {
        throw invalidModelError();
      }
      bucketsColumns = (0, _series.getBucketsColumns)(undefined, {
        split_mode: 'terms',
        terms_field: model.series[0].aggregate_by
      }, [], indexPattern, false);
      if (bucketsColumns === null) {
        throw invalidModelError();
      }
      columnStates.push((0, _table.getColumnState)(bucketsColumns[0].columnId, model.series[0].aggregate_function === 'mean' ? 'avg' : model.series[0].aggregate_function));
    }
    const metrics = [];

    // handle multiple layers/series
    for (const [_, series] of model.series.entries()) {
      if (series.hidden) {
        continue;
      }

      // not valid time shift
      if (series.offset_time && (0, _common.parseTimeShift)(series.offset_time) === 'invalid') {
        throw invalidModelError();
      }
      if (!(0, _metrics.isValidMetrics)(series.metrics, _enums.PANEL_TYPES.TABLE, series.time_range_mode)) {
        throw invalidModelError();
      }
      const reducedTimeRange = (0, _metrics.getReducedTimeRange)(model, series, timeRange);

      // handle multiple metrics
      const metricsColumns = (0, _series.getMetricsColumns)(series, indexPattern, seriesNum, {
        reducedTimeRange
      });
      if (!metricsColumns) {
        throw invalidModelError();
      }
      columnStates.push((0, _table.getColumnState)(metricsColumns[0].columnId, undefined, series));
      if (sortConfig.column === series.id) {
        sortConfiguration.columnId = metricsColumns[0].columnId;
      }
      metrics.push(...metricsColumns);
    }
    if (!metrics.length || metrics.every(metric => metric.operationType === 'static_value')) {
      throw invalidModelError();
    }
    const extendedLayer = {
      ignoreGlobalFilters: Boolean(model.ignore_global_filter ||
      // eslint-disable-next-line @typescript-eslint/naming-convention
      visibleSeries.some(({
        ignore_global_filter
      }) => ignore_global_filter)),
      indexPatternId: indexPatternId,
      layerId: (0, _uuid.v4)(),
      columns: [...metrics, ...commonBucketsColumns, ...bucketsColumns],
      columnOrder: []
    };
    const layers = Object.values(excludeMetaFromLayers({
      0: extendedLayer
    }));
    return {
      type: 'lnsDatatable',
      layers,
      configuration: {
        columns: columnStates,
        layerId: extendedLayer.layerId,
        layerType: 'data',
        sorting: sortConfiguration
      },
      indexPatternIds: (0, _convert_to_lens.getIndexPatternIds)(layers)
    };
  } catch (e) {
    return null;
  }
};
exports.convertToLens = convertToLens;