"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMetricsColumns = void 0;
var _series_agg = require("./series_agg");
var _metrics = require("../metrics");
var _convert = require("../convert");
var _columns = require("./columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getMetricsColumns = (series, dataView, visibleSeriesCount, {
  isStaticValueColumnSupported = false,
  reducedTimeRange
} = {}) => {
  const {
    metrics: validMetrics,
    seriesAgg
  } = (0, _series_agg.getSeriesAgg)(series.metrics);
  if (!validMetrics.length) {
    return null;
  }
  const metrics = validMetrics;
  // series agg supported as collapseFn if we have split
  if (seriesAgg && series.split_mode === 'everything') {
    return null;
  }
  const metricIdx = metrics.length - 1;
  const aggregation = metrics[metricIdx].type;
  const aggregationMap = _metrics.SUPPORTED_METRICS[aggregation];
  if (!aggregationMap) {
    return null;
  }
  const columnsConverterArgs = {
    series,
    metrics,
    dataView
  };
  switch (aggregation) {
    case 'percentile':
      {
        const percentileColumns = (0, _convert.convertToPercentileColumns)({
          series,
          metric: metrics[metricIdx],
          dataView
        }, {
          reducedTimeRange,
          timeShift: series.offset_time
        });
        return (0, _columns.getValidColumns)(percentileColumns);
      }
    case 'percentile_rank':
      {
        const percentileRankColumns = (0, _convert.convertToPercentileRankColumns)({
          series,
          metric: metrics[metricIdx],
          dataView
        }, {
          reducedTimeRange,
          timeShift: series.offset_time
        });
        return (0, _columns.getValidColumns)(percentileRankColumns);
      }
    case 'math':
      {
        const formulaColumn = (0, _convert.convertMathToFormulaColumn)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(formulaColumn);
      }
    case 'derivative':
    case 'moving_average':
      {
        const movingAverageOrDerivativeColumns = (0, _convert.convertParentPipelineAggToColumns)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(movingAverageOrDerivativeColumns);
      }
    case 'cumulative_sum':
      {
        const cumulativeSumColumns = (0, _convert.convertToCumulativeSumColumns)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(cumulativeSumColumns);
      }
    case 'filter_ratio':
      {
        const formulaColumn = (0, _convert.convertFilterRatioToFormulaColumn)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(formulaColumn);
      }
    case 'positive_rate':
      {
        const formulaColumn = (0, _convert.convertToCounterRateColumn)(columnsConverterArgs);
        return (0, _columns.getValidColumns)(formulaColumn);
      }
    case 'positive_only':
    case 'avg_bucket':
    case 'max_bucket':
    case 'min_bucket':
    case 'sum_bucket':
      {
        const formulaColumn = (0, _convert.convertOtherAggsToFormulaColumn)(aggregation, columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(formulaColumn);
      }
    case 'top_hit':
      {
        const column = (0, _convert.convertToLastValueColumn)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(column);
      }
    case 'variance':
      {
        const column = (0, _convert.convertVarianceToFormulaColumn)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(column);
      }
    case 'static':
      {
        const column = isStaticValueColumnSupported ? (0, _convert.convertToStaticValueColumn)(columnsConverterArgs, {
          visibleSeriesCount,
          reducedTimeRange
        }) : (0, _convert.convertStaticValueToFormulaColumn)(columnsConverterArgs);
        return (0, _columns.getValidColumns)(column);
      }
    case 'std_deviation':
      {
        const column = (0, _convert.convertToStandartDeviationColumn)(columnsConverterArgs, reducedTimeRange);
        return (0, _columns.getValidColumns)(column);
      }
    default:
      {
        const column = (0, _convert.convertMetricAggregationColumnWithoutSpecialParams)(aggregationMap, columnsConverterArgs, {
          reducedTimeRange,
          timeShift: series.offset_time
        });
        return (0, _columns.getValidColumns)(column);
      }
  }
};
exports.getMetricsColumns = getMetricsColumns;