"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaDeprecations = void 0;
var _react = _interopRequireWildcard(require("react"));
var _uuid = require("uuid");
var _reactRouterDom = require("react-router-dom");
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _analytics = require("@kbn/analytics");
var _shared_imports = require("../../../shared_imports");
var _app_context = require("../../app_context");
var _ui_metric = require("../../lib/ui_metric");
var _shared = require("../shared");
var _kibana_deprecations_table = require("./kibana_deprecations_table");
var _deprecation_details_flyout = require("./deprecation_details_flyout");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  useGlobalFlyout
} = _shared_imports.GlobalFlyout;
const i18nTexts = {
  pageTitle: _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.pageTitle', {
    defaultMessage: 'Kibana deprecation issues'
  }),
  pageDescription: _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.pageDescription', {
    defaultMessage: 'Resolve all critical issues before upgrading.'
  }),
  docLinkText: _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.docLinkText', {
    defaultMessage: 'Documentation'
  }),
  deprecationLabel: _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.deprecationLabel', {
    defaultMessage: 'Kibana'
  }),
  isLoading: _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.loadingText', {
    defaultMessage: 'Loading deprecation issues…'
  }),
  kibanaDeprecationErrorTitle: _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.kibanaDeprecationErrorTitle', {
    defaultMessage: 'List of deprecation issues might be incomplete'
  }),
  getKibanaDeprecationErrorDescription: pluginIds => _i18n.i18n.translate('xpack.upgradeAssistant.kibanaDeprecations.kibanaDeprecationErrorDescription', {
    defaultMessage: 'Failed to get deprecation issues for {pluginCount, plural, one {this plugin} other {these plugins}}: {pluginIds}. Check the Kibana server logs for more information.',
    values: {
      pluginCount: pluginIds.length,
      pluginIds: pluginIds.join(', ')
    }
  })
};
const getDeprecationCountByLevel = deprecations => {
  const criticalDeprecations = [];
  const warningDeprecations = [];
  deprecations.forEach(deprecation => {
    if (deprecation.level === 'critical') {
      criticalDeprecations.push(deprecation);
      return;
    }
    warningDeprecations.push(deprecation);
  });
  return {
    criticalDeprecations: criticalDeprecations.length,
    warningDeprecations: warningDeprecations.length
  };
};
const KibanaDeprecations = (0, _reactRouterDom.withRouter)(({
  history
}) => {
  const [kibanaDeprecations, setKibanaDeprecations] = (0, _react.useState)(undefined);
  const [kibanaDeprecationErrors, setKibanaDeprecationErrors] = (0, _react.useState)([]);
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const [error, setError] = (0, _react.useState)(undefined);
  const [flyoutContent, setFlyoutContent] = (0, _react.useState)(undefined);
  const [deprecationResolutionState, setDeprecationResolutionState] = (0, _react.useState)(undefined);
  const {
    services: {
      core: {
        deprecations
      },
      breadcrumbs
    }
  } = (0, _app_context.useAppContext)();
  const {
    addContent: addContentToGlobalFlyout,
    removeContent: removeContentFromGlobalFlyout
  } = useGlobalFlyout();
  const getAllDeprecations = (0, _react.useCallback)(async () => {
    setIsLoading(true);
    try {
      const allDeprecations = await deprecations.getAllDeprecations();
      const filteredDeprecations = [];
      const deprecationErrors = [];
      allDeprecations.forEach(deprecation => {
        var _deprecation$deprecat;
        // Keep track of any plugin deprecations that failed to fetch to show warning in UI
        if (deprecation.level === 'fetch_error') {
          // It's possible that a plugin registered more than one deprecation that could fail
          // We only want to keep track of the unique plugin failures
          const pluginErrorExists = deprecationErrors.includes(deprecation.domainId);
          if (pluginErrorExists === false) {
            deprecationErrors.push(deprecation.domainId);
          }
          return;
        }

        // Only show deprecations in the table that fetched successfully
        filteredDeprecations.push({
          ...deprecation,
          id: (0, _uuid.v4)(),
          // Associate an unique ID with each deprecation to track resolution state
          filterType: (_deprecation$deprecat = deprecation.deprecationType) !== null && _deprecation$deprecat !== void 0 ? _deprecation$deprecat : 'uncategorized' // deprecationType is currently optional, in order to correctly handle sort/filter, we default any undefined types to "uncategorized"
        });
      });

      setKibanaDeprecations(filteredDeprecations);
      setKibanaDeprecationErrors(deprecationErrors);
    } catch (e) {
      setError(e);
    }
    setIsLoading(false);
  }, [deprecations]);
  const deprecationsCountByLevel = (0, _react.useMemo)(() => getDeprecationCountByLevel(kibanaDeprecations || []), [kibanaDeprecations]);
  const toggleFlyout = newFlyoutContent => {
    setFlyoutContent(newFlyoutContent);
  };
  const closeFlyout = (0, _react.useCallback)(() => {
    toggleFlyout();
    removeContentFromGlobalFlyout('deprecationDetails');
  }, [removeContentFromGlobalFlyout]);
  const resolveDeprecation = (0, _react.useCallback)(async deprecationDetails => {
    setDeprecationResolutionState({
      id: deprecationDetails.id,
      resolveDeprecationStatus: 'in_progress'
    });
    const response = await deprecations.resolveDeprecation(deprecationDetails);
    setDeprecationResolutionState({
      id: deprecationDetails.id,
      resolveDeprecationStatus: response.status,
      resolveDeprecationError: response.status === 'fail' ? response.reason : undefined
    });
    closeFlyout();
  }, [closeFlyout, deprecations]);
  (0, _react.useEffect)(() => {
    if (flyoutContent) {
      addContentToGlobalFlyout({
        id: 'deprecationDetails',
        Component: _deprecation_details_flyout.DeprecationDetailsFlyout,
        props: {
          deprecation: flyoutContent,
          closeFlyout,
          resolveDeprecation,
          deprecationResolutionState: deprecationResolutionState && flyoutContent.id === deprecationResolutionState.id ? deprecationResolutionState : undefined
        },
        flyoutProps: {
          onClose: closeFlyout,
          'data-test-subj': 'kibanaDeprecationDetails',
          'aria-labelledby': 'kibanaDeprecationDetailsFlyoutTitle'
        }
      });
    }
  }, [addContentToGlobalFlyout, closeFlyout, deprecationResolutionState, flyoutContent, resolveDeprecation]);
  (0, _react.useEffect)(() => {
    _ui_metric.uiMetricService.trackUiMetric(_analytics.METRIC_TYPE.LOADED, _ui_metric.UIM_KIBANA_DEPRECATIONS_PAGE_LOAD);
  }, []);
  (0, _react.useEffect)(() => {
    breadcrumbs.setBreadcrumbs('kibanaDeprecations');
  }, [breadcrumbs]);
  (0, _react.useEffect)(() => {
    getAllDeprecations();
  }, [deprecations, getAllDeprecations]);
  if (error) {
    return /*#__PURE__*/_react.default.createElement(_shared.DeprecationsPageLoadingError, {
      deprecationSource: "Kibana"
    });
  }
  if (isLoading) {
    return /*#__PURE__*/_react.default.createElement(_shared_imports.SectionLoading, null, i18nTexts.isLoading);
  }
  if ((kibanaDeprecations === null || kibanaDeprecations === void 0 ? void 0 : kibanaDeprecations.length) === 0) {
    return /*#__PURE__*/_react.default.createElement(_shared.NoDeprecationsPrompt, {
      deprecationType: i18nTexts.deprecationLabel,
      navigateToOverviewPage: () => history.push('/overview')
    });
  }
  return /*#__PURE__*/_react.default.createElement("div", {
    "data-test-subj": "kibanaDeprecations"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiPageHeader, {
    bottomBorder: true,
    pageTitle: i18nTexts.pageTitle,
    description: i18nTexts.pageDescription
  }, /*#__PURE__*/_react.default.createElement(_shared.DeprecationCount, {
    totalCriticalDeprecations: deprecationsCountByLevel.criticalDeprecations,
    totalWarningDeprecations: deprecationsCountByLevel.warningDeprecations
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "l"
  }), kibanaDeprecationErrors.length > 0 && /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
    title: i18nTexts.kibanaDeprecationErrorTitle,
    color: "warning",
    iconType: "warning",
    "data-test-subj": "kibanaDeprecationErrors"
  }, /*#__PURE__*/_react.default.createElement("p", null, i18nTexts.getKibanaDeprecationErrorDescription(kibanaDeprecationErrors))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, null)), /*#__PURE__*/_react.default.createElement(_kibana_deprecations_table.KibanaDeprecationsTable, {
    deprecations: kibanaDeprecations,
    reload: getAllDeprecations,
    toggleFlyout: toggleFlyout,
    deprecationResolutionState: deprecationResolutionState
  }));
});
exports.KibanaDeprecations = KibanaDeprecations;