"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFilterAggConfig = getFilterAggConfig;
exports.getFilterAggTypeConfig = getFilterAggTypeConfig;
var _validators = require("../../../../../../common/validators");
var _pivot_aggs = require("../../../../../../common/pivot_aggs");
var _constants = require("./constants");
var _components = require("./components");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Gets initial basic configuration of the filter aggregation.
 */
function getFilterAggConfig(commonConfig) {
  const field = (0, _pivot_aggs.isPivotAggsConfigWithUiBase)(commonConfig) ? commonConfig.field : null;
  return {
    ...commonConfig,
    isSubAggsSupported: true,
    // Field name might be missing, for instance for the bool filter.
    field,
    AggFormComponent: _components.FilterAggForm,
    aggConfig: {},
    getEsAggConfig() {
      var _this$aggConfig$aggTy;
      // ensure the configuration has been completed
      if (!this.isValid()) {
        return null;
      }
      const esAgg = (_this$aggConfig$aggTy = this.aggConfig.aggTypeConfig) === null || _this$aggConfig$aggTy === void 0 ? void 0 : _this$aggConfig$aggTy.getEsAggConfig(this.field);
      return {
        [this.aggConfig.filterAgg]: esAgg
      };
    },
    setUiConfigFromEs(esAggDefinition) {
      var _ref;
      const filterAgg = Object.keys(esAggDefinition)[0];
      // @ts-ignore conflicts with a union type
      const filterAggConfig = esAggDefinition[filterAgg];
      const aggTypeConfig = getFilterAggTypeConfig(filterAgg, field, filterAggConfig);
      this.field = (_ref = field !== null && field !== void 0 ? field : aggTypeConfig.fieldName) !== null && _ref !== void 0 ? _ref : null;
      this.aggConfig = {
        filterAgg,
        aggTypeConfig
      };
    },
    isValid() {
      var _this$aggConfig, _this$aggConfig$aggTy2;
      return ((_this$aggConfig = this.aggConfig) === null || _this$aggConfig === void 0 ? void 0 : _this$aggConfig.filterAgg) !== undefined && ((_this$aggConfig$aggTy2 = this.aggConfig.aggTypeConfig) !== null && _this$aggConfig$aggTy2 !== void 0 && _this$aggConfig$aggTy2.isValid ? this.aggConfig.aggTypeConfig.isValid() : true);
    },
    getAggName() {
      var _this$aggConfig2, _this$aggConfig2$aggT;
      return (_this$aggConfig2 = this.aggConfig) !== null && _this$aggConfig2 !== void 0 && (_this$aggConfig2$aggT = _this$aggConfig2.aggTypeConfig) !== null && _this$aggConfig2$aggT !== void 0 && _this$aggConfig2$aggT.getAggName ? this.aggConfig.aggTypeConfig.getAggName() : undefined;
    },
    helperText() {
      var _this$aggConfig3, _this$aggConfig3$aggT;
      return (_this$aggConfig3 = this.aggConfig) !== null && _this$aggConfig3 !== void 0 && (_this$aggConfig3$aggT = _this$aggConfig3.aggTypeConfig) !== null && _this$aggConfig3$aggT !== void 0 && _this$aggConfig3$aggT.helperText ? this.aggConfig.aggTypeConfig.helperText() : undefined;
    }
  };
}

/**
 * Returns a form component for provided filter aggregation type.
 */
function getFilterAggTypeConfig(filterAggType, fieldName, esConfig) {
  var _esFilterRange$gte, _esFilterRange$lte;
  let resultField = fieldName;
  switch (filterAggType) {
    case _constants.FILTERS.TERM:
      const value = typeof esConfig === 'object' ? Object.values(esConfig)[0] : undefined;
      resultField = esConfig ? Object.keys(esConfig)[0] : resultField;
      return {
        FilterAggFormComponent: _components.FilterTermForm,
        filterAggConfig: {
          value
        },
        getEsAggConfig() {
          if (this.fieldName === undefined || !this.filterAggConfig) {
            throw new Error(`Config ${_constants.FILTERS.TERM} is not completed`);
          }
          return {
            [this.fieldName]: this.filterAggConfig.value
          };
        },
        isValid() {
          var _this$filterAggConfig;
          return ((_this$filterAggConfig = this.filterAggConfig) === null || _this$filterAggConfig === void 0 ? void 0 : _this$filterAggConfig.value) !== undefined;
        },
        getAggName() {
          var _this$filterAggConfig2;
          return (_this$filterAggConfig2 = this.filterAggConfig) !== null && _this$filterAggConfig2 !== void 0 && _this$filterAggConfig2.value ? this.filterAggConfig.value : undefined;
        },
        fieldName: resultField
      };
    case _constants.FILTERS.RANGE:
      resultField = esConfig ? Object.keys(esConfig)[0] : resultField;
      const esFilterRange = typeof esConfig === 'object' ? Object.values(esConfig)[0] : undefined;
      return {
        fieldName: resultField,
        FilterAggFormComponent: _components.FilterRangeForm,
        filterAggConfig: typeof esFilterRange === 'object' ? {
          from: (_esFilterRange$gte = esFilterRange.gte) !== null && _esFilterRange$gte !== void 0 ? _esFilterRange$gte : esFilterRange.gt,
          to: (_esFilterRange$lte = esFilterRange.lte) !== null && _esFilterRange$lte !== void 0 ? _esFilterRange$lte : esFilterRange.lt,
          includeFrom: esFilterRange.gte !== undefined,
          includeTo: esFilterRange.lte !== undefined
        } : undefined,
        getEsAggConfig() {
          if (this.fieldName === undefined || !this.filterAggConfig) {
            throw new Error(`Config ${_constants.FILTERS.RANGE} is not completed`);
          }
          const {
            from,
            includeFrom,
            to,
            includeTo
          } = this.filterAggConfig;
          const result = {};
          if (from) {
            result[includeFrom ? 'gte' : 'gt'] = from;
          }
          if (to) {
            result[includeTo ? 'lte' : 'lt'] = to;
          }
          return {
            [this.fieldName]: result
          };
        },
        isValid() {
          if (typeof this.filterAggConfig !== 'object' || this.filterAggConfig.from === undefined && this.filterAggConfig.to === undefined) {
            return false;
          }
          if (this.filterAggConfig.from !== undefined && this.filterAggConfig.to !== undefined) {
            return this.filterAggConfig.from <= this.filterAggConfig.to;
          }
          return true;
        },
        helperText() {
          if (!this.isValid()) return;
          const {
            from,
            to,
            includeFrom,
            includeTo
          } = this.filterAggConfig;
          return `range: ${`${from !== undefined ? `${includeFrom ? '≥' : '>'} ${from}` : ''} ${from !== undefined && to !== undefined ? '&' : ''} ${to !== undefined ? `${includeTo ? '≤' : '<'} ${to}` : ''}`.trim()}`;
        }
      };
    case _constants.FILTERS.EXISTS:
      resultField = esConfig ? esConfig.field : resultField;
      return {
        fieldName: resultField,
        getEsAggConfig() {
          if (this.fieldName === undefined) {
            throw new Error(`Config ${_constants.FILTERS.EXISTS} is not completed`);
          }
          return {
            field: this.fieldName
          };
        },
        isValid() {
          return typeof this.fieldName === 'string';
        }
      };
    case _constants.FILTERS.BOOL:
      return {
        FilterAggFormComponent: _components.FilterEditorForm,
        filterAggConfig: JSON.stringify({
          must: [],
          must_not: [],
          should: []
        }, null, 2),
        isValid() {
          return (0, _validators.jsonStringValidator)(this.filterAggConfig);
        },
        getEsAggConfig() {
          return JSON.parse(this.filterAggConfig);
        }
      };
    default:
      return {
        fieldName,
        FilterAggFormComponent: _components.FilterEditorForm,
        filterAggConfig: '',
        getEsAggConfig() {
          return this.filterAggConfig !== undefined ? JSON.parse(this.filterAggConfig) : {};
        },
        isValid() {
          return (0, _validators.jsonStringValidator)(this.filterAggConfig);
        }
      };
  }
}