"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskClaiming = exports.TASK_MANAGER_MARK_AS_CLAIMED = exports.BatchConcurrency = void 0;
exports.isClaimOwnershipResult = isClaimOwnershipResult;
exports.isLimited = isLimited;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _lodash = require("lodash");
var _result_type = require("../lib/result_type");
var _fill_pool = require("../lib/fill_pool");
var _task_claimers = require("../task_claimers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * This module contains helpers for managing the task manager storage layer.
 */

function isClaimOwnershipResult(result) {
  return (0, _lodash.isPlainObject)(result.stats) && Array.isArray(result.docs);
}
let BatchConcurrency;
exports.BatchConcurrency = BatchConcurrency;
(function (BatchConcurrency) {
  BatchConcurrency[BatchConcurrency["Unlimited"] = 0] = "Unlimited";
  BatchConcurrency[BatchConcurrency["Limited"] = 1] = "Limited";
})(BatchConcurrency || (exports.BatchConcurrency = BatchConcurrency = {}));
const TASK_MANAGER_MARK_AS_CLAIMED = 'mark-available-tasks-as-claimed';
exports.TASK_MANAGER_MARK_AS_CLAIMED = TASK_MANAGER_MARK_AS_CLAIMED;
class TaskClaiming {
  /**
   * Constructs a new TaskStore.
   * @param {TaskClaimingOpts} opts
   * @prop {number} maxAttempts - The maximum number of attempts before a task will be abandoned
   * @prop {TaskDefinition} definition - The definition of the task being run
   */
  constructor(opts) {
    (0, _defineProperty2.default)(this, "errors$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "maxAttempts", void 0);
    (0, _defineProperty2.default)(this, "definitions", void 0);
    (0, _defineProperty2.default)(this, "events$", void 0);
    (0, _defineProperty2.default)(this, "taskStore", void 0);
    (0, _defineProperty2.default)(this, "getCapacity", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskClaimingBatchesByType", void 0);
    (0, _defineProperty2.default)(this, "taskMaxAttempts", void 0);
    (0, _defineProperty2.default)(this, "excludedTaskTypes", void 0);
    (0, _defineProperty2.default)(this, "unusedTypes", void 0);
    (0, _defineProperty2.default)(this, "taskClaimer", void 0);
    (0, _defineProperty2.default)(this, "claimingBatchIndex", 0);
    this.definitions = opts.definitions;
    this.maxAttempts = opts.maxAttempts;
    this.taskStore = opts.taskStore;
    this.getCapacity = opts.getCapacity;
    this.logger = opts.logger.get('taskClaiming');
    this.taskClaimingBatchesByType = this.partitionIntoClaimingBatches(this.definitions);
    this.taskMaxAttempts = Object.fromEntries(this.normalizeMaxAttempts(this.definitions));
    this.excludedTaskTypes = opts.excludedTaskTypes;
    this.unusedTypes = opts.unusedTypes;
    this.taskClaimer = (0, _task_claimers.getTaskClaimer)(opts.strategy);
    this.events$ = new _rxjs.Subject();
  }
  partitionIntoClaimingBatches(definitions) {
    const {
      limitedConcurrency,
      unlimitedConcurrency,
      skippedTypes
    } = (0, _lodash.groupBy)(definitions.getAllDefinitions(), definition => definition.maxConcurrency ? 'limitedConcurrency' : definition.maxConcurrency === 0 ? 'skippedTypes' : 'unlimitedConcurrency');
    if (skippedTypes !== null && skippedTypes !== void 0 && skippedTypes.length) {
      this.logger.info(`Task Manager will never claim tasks of the following types as their "maxConcurrency" is set to 0: ${skippedTypes.map(({
        type
      }) => type).join(', ')}`);
    }
    return [...(unlimitedConcurrency ? [asUnlimited(new Set(unlimitedConcurrency.map(({
      type
    }) => type)))] : []), ...(limitedConcurrency ? limitedConcurrency.map(({
      type
    }) => asLimited(type)) : [])];
  }
  normalizeMaxAttempts(definitions) {
    return new Map([...definitions].map(([type, {
      maxAttempts
    }]) => [type, maxAttempts || this.maxAttempts]));
  }
  getClaimingBatches() {
    // return all batches, starting at index and cycling back to where we began
    const batch = [...this.taskClaimingBatchesByType.slice(this.claimingBatchIndex), ...this.taskClaimingBatchesByType.slice(0, this.claimingBatchIndex)];
    // shift claimingBatchIndex by one so that next cycle begins at the next index
    this.claimingBatchIndex = (this.claimingBatchIndex + 1) % this.taskClaimingBatchesByType.length;
    return batch;
  }
  get events() {
    return this.events$;
  }
  claimAvailableTasksIfCapacityIsAvailable(claimingOptions) {
    if (this.getCapacity()) {
      const opts = {
        batches: this.getClaimingBatches(),
        claimOwnershipUntil: claimingOptions.claimOwnershipUntil,
        taskStore: this.taskStore,
        events$: this.events$,
        getCapacity: this.getCapacity,
        unusedTypes: this.unusedTypes,
        definitions: this.definitions,
        taskMaxAttempts: this.taskMaxAttempts,
        excludedTaskTypes: this.excludedTaskTypes
      };
      return this.taskClaimer(opts).pipe((0, _operators.map)(claimResult => (0, _result_type.asOk)(claimResult)));
    }
    this.logger.debug(`[Task Ownership]: Task Manager has skipped Claiming Ownership of available tasks at it has ran out Available Workers.`);
    return (0, _rxjs.of)((0, _result_type.asErr)(_fill_pool.FillPoolResult.NoAvailableWorkers));
  }
}
exports.TaskClaiming = TaskClaiming;
function isLimited(batch) {
  return batch.concurrency === BatchConcurrency.Limited;
}
function asLimited(tasksType) {
  return {
    concurrency: BatchConcurrency.Limited,
    tasksTypes: tasksType
  };
}
function asUnlimited(tasksTypes) {
  return {
    concurrency: BatchConcurrency.Unlimited,
    tasksTypes
  };
}