"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processMonitors = exports.getAllMonitors = void 0;
var _lodash = require("lodash");
var _saved_objects = require("../../../common/types/saved_objects");
var _overview_status = require("../../routes/overview_status/overview_status");
var _runtime_types = require("../../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAllMonitors = async ({
  soClient,
  search,
  fields,
  filter,
  sortField = 'name.keyword',
  sortOrder = 'asc',
  searchFields
}) => {
  const finder = soClient.createPointInTimeFinder({
    type: _saved_objects.syntheticsMonitorType,
    perPage: 1000,
    search,
    sortField,
    sortOrder,
    fields,
    filter,
    searchFields
  });
  const hits = [];
  for await (const result of finder.find()) {
    hits.push(...result.saved_objects);
  }

  // no need to wait for it
  finder.close();
  return hits;
};
exports.getAllMonitors = getAllMonitors;
const processMonitors = (allMonitors, server, soClient, syntheticsMonitorClient, queryLocations) => {
  /**
   * Walk through all monitor saved objects, bucket IDs by disabled/enabled status.
   *
   * Track max period to make sure the snapshot query should reach back far enough to catch
   * latest ping for all enabled monitors.
   */
  const enabledMonitorQueryIds = [];
  const disabledMonitorQueryIds = [];
  let disabledCount = 0;
  let disabledMonitorsCount = 0;
  let maxPeriod = 0;
  let projectMonitorsCount = 0;
  const allIds = [];
  let listOfLocationsSet = new Set();
  const monitorLocationMap = {};
  const monitorQueryIdToConfigIdMap = {};
  for (const monitor of allMonitors) {
    const attrs = monitor.attributes;
    allIds.push(attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]);
    projectMonitorsCount += (attrs === null || attrs === void 0 ? void 0 : attrs[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE]) === _runtime_types.SourceType.PROJECT ? 1 : 0;
    monitorQueryIdToConfigIdMap[attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]] = attrs[_runtime_types.ConfigKey.CONFIG_ID];
    const monitorLocations = attrs[_runtime_types.ConfigKey.LOCATIONS].map(location => location.id);
    if (attrs[_runtime_types.ConfigKey.ENABLED] === false) {
      const queriedLocations = Array.isArray(queryLocations) ? queryLocations : [queryLocations];
      const intersectingLocations = (0, _lodash.intersection)(monitorLocations, queryLocations ? queriedLocations : monitorLocations);
      disabledCount += intersectingLocations.length;
      disabledMonitorsCount += 1;
      disabledMonitorQueryIds.push(attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]);
    } else {
      enabledMonitorQueryIds.push(attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]);
      monitorLocationMap[attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]] = queryLocations ? (0, _lodash.intersection)(monitorLocations, queryLocations) : monitorLocations;
      listOfLocationsSet = new Set([...listOfLocationsSet, ...monitorLocations]);
      maxPeriod = Math.max(maxPeriod, (0, _overview_status.periodToMs)(attrs[_runtime_types.ConfigKey.SCHEDULE]));
    }
  }
  return {
    maxPeriod,
    allIds,
    enabledMonitorQueryIds,
    disabledMonitorQueryIds,
    disabledCount,
    monitorLocationMap,
    disabledMonitorsCount,
    projectMonitorsCount,
    monitorLocationIds: [...listOfLocationsSet],
    monitorQueryIdToConfigIdMap
  };
};
exports.processMonitors = processMonitors;