"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validatePermissions = exports.syncEditedMonitor = exports.editSyntheticsMonitorRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _common = require("@kbn/spaces-plugin/common");
var _add_monitor_project = require("./add_monitor_project");
var _synthetics_monitor = require("../../saved_objects/synthetics_monitor");
var _get_private_locations = require("../../synthetics_service/get_private_locations");
var _helper = require("./helper");
var _saved_objects = require("../../../common/types/saved_objects");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _monitor_validation = require("./monitor_validation");
var _service_errors = require("../synthetics_service/service_errors");
var _monitor_upgrade_sender = require("../telemetry/monitor_upgrade_sender");
var _secrets = require("../../synthetics_service/utils/secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Simplify return promise type and type it with runtime_types
const editSyntheticsMonitorRoute = () => ({
  method: 'PUT',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_MONITORS + '/{monitorId}',
  validate: {
    params: _configSchema.schema.object({
      monitorId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.any()
  },
  handler: async routeContext => {
    const {
      request,
      response,
      savedObjectsClient,
      server
    } = routeContext;
    const {
      logger
    } = server;
    const monitor = request.body;
    const {
      monitorId
    } = request.params;
    try {
      var _server$spaces$spaces, _server$spaces, _previousMonitor$name;
      const spaceId = (_server$spaces$spaces = (_server$spaces = server.spaces) === null || _server$spaces === void 0 ? void 0 : _server$spaces.spacesService.getSpaceId(request)) !== null && _server$spaces$spaces !== void 0 ? _server$spaces$spaces : _common.DEFAULT_SPACE_ID;
      const previousMonitor = await savedObjectsClient.get(_saved_objects.syntheticsMonitorType, monitorId);

      /* Decrypting the previous monitor before editing ensures that all existing fields remain
       * on the object, even in flows where decryption does not take place, such as the enabled tab
       * on the monitor list table. We do not decrypt monitors in bulk for the monitor list table */
      const decryptedPreviousMonitor = await (0, _synthetics_monitor.getDecryptedMonitor)(server, monitorId, (_previousMonitor$name = previousMonitor.namespaces) === null || _previousMonitor$name === void 0 ? void 0 : _previousMonitor$name[0]);
      const normalizedPreviousMonitor = (0, _secrets.normalizeSecrets)(decryptedPreviousMonitor).attributes;
      const editedMonitor = (0, _helper.mergeSourceMonitor)(normalizedPreviousMonitor, monitor);
      const validationResult = (0, _monitor_validation.validateMonitor)(editedMonitor);
      if (!validationResult.valid || !validationResult.decodedMonitor) {
        const {
          reason: message,
          details,
          payload
        } = validationResult;
        return response.badRequest({
          body: {
            message,
            attributes: {
              details,
              ...payload
            }
          }
        });
      }
      const err = await validatePermissions(routeContext, editedMonitor.locations);
      if (err) {
        return response.forbidden({
          body: {
            message: err
          }
        });
      }
      const monitorWithRevision = {
        ...validationResult.decodedMonitor,
        /* reset config hash to empty string. Ensures that the synthetics agent is able
         * to update project monitors on when next pushed after they are edited via the UI,
         * through the enable/disable monitor toggle */
        [_runtime_types.ConfigKey.CONFIG_HASH]: '',
        revision: (previousMonitor.attributes[_runtime_types.ConfigKey.REVISION] || 0) + 1
      };
      const {
        publicSyncErrors,
        failedPolicyUpdates,
        editedMonitor: editedMonitorSavedObject
      } = await syncEditedMonitor({
        routeContext,
        previousMonitor,
        decryptedPreviousMonitor,
        normalizedMonitor: monitorWithRevision,
        spaceId
      });
      if (failedPolicyUpdates && failedPolicyUpdates.length > 0) {
        const hasError = failedPolicyUpdates.find(update => update.error);
        await rollbackUpdate({
          routeContext,
          configId: monitorId,
          attributes: decryptedPreviousMonitor.attributes
        });
        throw hasError === null || hasError === void 0 ? void 0 : hasError.error;
      }

      // Return service sync errors in OK response
      if (publicSyncErrors && publicSyncErrors.length > 0) {
        return response.ok({
          body: {
            message: 'error pushing monitor to the service',
            attributes: {
              errors: publicSyncErrors
            }
          }
        });
      }
      return (0, _helper.mapSavedObjectToMonitor)(editedMonitorSavedObject);
    } catch (updateErr) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(updateErr)) {
        return (0, _service_errors.getMonitorNotFoundResponse)(response, monitorId);
      }
      logger.error(updateErr);
      return response.customError({
        body: {
          message: updateErr.message
        },
        statusCode: 500
      });
    }
  }
});
exports.editSyntheticsMonitorRoute = editSyntheticsMonitorRoute;
const rollbackUpdate = async ({
  routeContext,
  configId,
  attributes
}) => {
  const {
    savedObjectsClient,
    server
  } = routeContext;
  try {
    await savedObjectsClient.update(_saved_objects.syntheticsMonitorType, configId, attributes);
  } catch (e) {
    server.logger.error(`Unable to rollback Synthetics monitors edit ${e.message} `);
  }
};
const syncEditedMonitor = async ({
  normalizedMonitor,
  previousMonitor,
  decryptedPreviousMonitor,
  spaceId,
  routeContext
}) => {
  const {
    server,
    savedObjectsClient,
    syntheticsMonitorClient
  } = routeContext;
  try {
    const monitorWithId = {
      ...normalizedMonitor,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || previousMonitor.id,
      [_runtime_types.ConfigKey.CONFIG_ID]: previousMonitor.id
    };
    const formattedMonitor = (0, _secrets.formatSecrets)(monitorWithId);
    const editedSOPromise = savedObjectsClient.update(_saved_objects.syntheticsMonitorType, previousMonitor.id, formattedMonitor);
    const allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(savedObjectsClient);
    const editSyncPromise = syntheticsMonitorClient.editMonitors([{
      monitor: monitorWithId,
      id: previousMonitor.id,
      previousMonitor,
      decryptedPreviousMonitor
    }], routeContext, allPrivateLocations, spaceId);
    const [editedMonitorSavedObject, {
      publicSyncErrors,
      failedPolicyUpdates
    }] = await Promise.all([editedSOPromise, editSyncPromise]).catch(e => {
      server.logger.error(e);
      throw e;
    });
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryUpdateEvent)(editedMonitorSavedObject, previousMonitor, server.stackVersion, Boolean(normalizedMonitor[_runtime_types.ConfigKey.SOURCE_INLINE]), publicSyncErrors));
    return {
      failedPolicyUpdates,
      publicSyncErrors,
      editedMonitor: editedMonitorSavedObject
    };
  } catch (e) {
    server.logger.error(`Unable to update Synthetics monitor ${decryptedPreviousMonitor.attributes[_runtime_types.ConfigKey.NAME]}`);
    await rollbackUpdate({
      routeContext,
      configId: previousMonitor.id,
      attributes: decryptedPreviousMonitor.attributes
    });
    throw e;
  }
};
exports.syncEditedMonitor = syncEditedMonitor;
const validatePermissions = async ({
  server,
  response,
  request
}, monitorLocations) => {
  var _Boolean, _server$coreStart;
  const hasPublicLocations = monitorLocations === null || monitorLocations === void 0 ? void 0 : monitorLocations.some(loc => loc.isServiceManaged);
  if (!hasPublicLocations) {
    return;
  }
  const elasticManagedLocationsEnabled = (_Boolean = Boolean((await ((_server$coreStart = server.coreStart) === null || _server$coreStart === void 0 ? void 0 : _server$coreStart.capabilities.resolveCapabilities(request, {
    capabilityPath: 'uptime.*'
  }))).uptime.elasticManagedLocationsEnabled)) !== null && _Boolean !== void 0 ? _Boolean : true;
  if (!elasticManagedLocationsEnabled) {
    return _add_monitor_project.ELASTIC_MANAGED_LOCATIONS_DISABLED;
  }
};
exports.validatePermissions = validatePermissions;