"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteSyntheticsMonitorRoute = exports.deleteMonitor = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _edit_monitor = require("./edit_monitor");
var _saved_objects = require("../../../common/types/saved_objects");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _service_errors = require("../synthetics_service/service_errors");
var _monitor_upgrade_sender = require("../telemetry/monitor_upgrade_sender");
var _secrets = require("../../synthetics_service/utils/secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteSyntheticsMonitorRoute = () => ({
  method: 'DELETE',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_MONITORS + '/{monitorId}',
  validate: {
    params: _configSchema.schema.object({
      monitorId: _configSchema.schema.string({
        minLength: 1,
        maxLength: 1024
      })
    })
  },
  handler: async routeContext => {
    const {
      request,
      response
    } = routeContext;
    const {
      monitorId
    } = request.params;
    try {
      const {
        errors,
        res
      } = await deleteMonitor({
        routeContext,
        monitorId
      });
      if (res) {
        return res;
      }
      if (errors && errors.length > 0) {
        return response.ok({
          body: {
            message: 'error pushing monitor to the service',
            attributes: {
              errors
            }
          }
        });
      }
      return monitorId;
    } catch (getErr) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(getErr)) {
        return (0, _service_errors.getMonitorNotFoundResponse)(response, monitorId);
      }
      throw getErr;
    }
  }
});
exports.deleteSyntheticsMonitorRoute = deleteSyntheticsMonitorRoute;
const deleteMonitor = async ({
  routeContext,
  monitorId
}) => {
  const {
    response,
    spaceId,
    savedObjectsClient,
    server,
    syntheticsMonitorClient
  } = routeContext;
  const {
    logger,
    telemetry,
    stackVersion
  } = server;
  const {
    monitor,
    monitorWithSecret
  } = await getMonitorToDelete(monitorId, savedObjectsClient, server, spaceId);
  const err = await (0, _edit_monitor.validatePermissions)(routeContext, monitor.attributes.locations);
  if (err) {
    return {
      res: response.forbidden({
        body: {
          message: err
        }
      })
    };
  }
  let deletePromise;
  try {
    deletePromise = savedObjectsClient.delete(_saved_objects.syntheticsMonitorType, monitorId);
    const deleteSyncPromise = syntheticsMonitorClient.deleteMonitors([{
      ...monitor.attributes,
      id: monitor.attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID]
    }
    /* Type cast encrypted saved objects to decrypted saved objects for delete flow only.
     * Deletion does not require all monitor fields */], savedObjectsClient, spaceId);
    const [errors] = await Promise.all([deleteSyncPromise, deletePromise]).catch(e => {
      server.logger.error(e);
      throw e;
    });
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(logger, telemetry, (0, _monitor_upgrade_sender.formatTelemetryDeleteEvent)(monitor, stackVersion, new Date().toISOString(), Boolean(monitor.attributes[_runtime_types.ConfigKey.SOURCE_INLINE]), errors));
    return {
      errors
    };
  } catch (e) {
    if (deletePromise) {
      await deletePromise;
    }
    server.logger.error(`Unable to delete Synthetics monitor ${monitor.attributes[_runtime_types.ConfigKey.NAME]}`);
    if (monitorWithSecret) {
      await restoreDeletedMonitor({
        monitorId,
        normalizedMonitor: (0, _secrets.formatSecrets)({
          ...monitorWithSecret.attributes
        }),
        savedObjectsClient
      });
    }
    throw e;
  }
};
exports.deleteMonitor = deleteMonitor;
const getMonitorToDelete = async (monitorId, soClient, server, spaceId) => {
  const encryptedSOClient = server.encryptedSavedObjects.getClient();
  try {
    const monitor = await encryptedSOClient.getDecryptedAsInternalUser(_saved_objects.syntheticsMonitorType, monitorId, {
      namespace: spaceId
    });
    return {
      monitor: (0, _secrets.normalizeSecrets)(monitor),
      monitorWithSecret: (0, _secrets.normalizeSecrets)(monitor)
    };
  } catch (e) {
    server.logger.error(`Failed to decrypt monitor to delete ${monitorId}${e}`);
    (0, _monitor_upgrade_sender.sendErrorTelemetryEvents)(server.logger, server.telemetry, {
      reason: `Failed to decrypt monitor to delete ${monitorId}`,
      message: e === null || e === void 0 ? void 0 : e.message,
      type: 'deletionError',
      code: e === null || e === void 0 ? void 0 : e.code,
      status: e.status,
      stackVersion: server.stackVersion
    });
  }
  const monitor = await soClient.get(_saved_objects.syntheticsMonitorType, monitorId);
  return {
    monitor,
    withSecrets: false
  };
};
const restoreDeletedMonitor = async ({
  monitorId,
  savedObjectsClient,
  normalizedMonitor
}) => {
  try {
    await savedObjectsClient.get(_saved_objects.syntheticsMonitorType, monitorId);
  } catch (e) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      await savedObjectsClient.create(_saved_objects.syntheticsMonitorType, normalizedMonitor, {
        id: monitorId,
        overwrite: true
      });
    }
  }
};