"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenAIConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _stream = require("stream");
var _schema = require("../../../common/openai/schema");
var _constants = require("../../../common/openai/constants");
var _create_gen_ai_dashboard = require("../lib/gen_ai/create_gen_ai_dashboard");
var _utils = require("./lib/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class OpenAIConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "url", void 0);
    (0, _defineProperty2.default)(this, "provider", void 0);
    (0, _defineProperty2.default)(this, "key", void 0);
    this.url = this.config.apiUrl;
    this.provider = this.config.apiProvider;
    this.key = this.secrets.apiKey;
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.SUB_ACTION.RUN,
      method: 'runApi',
      schema: _schema.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.TEST,
      method: 'runApi',
      schema: _schema.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.STREAM,
      method: 'streamApi',
      schema: _schema.StreamActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.DASHBOARD,
      method: 'getDashboard',
      schema: _schema.DashboardActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.INVOKE_AI,
      method: 'invokeAI',
      schema: _schema.InvokeAIActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.INVOKE_STREAM,
      method: 'invokeStream',
      schema: _schema.InvokeAIActionParamsSchema
    });
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response3, _error$response4, _error$response4$data, _error$response4$data2, _error$response$data$2;
    if (!((_error$response = error.response) !== null && _error$response !== void 0 && _error$response.status)) {
      var _error$code, _error$message;
      return `Unexpected API Error: ${(_error$code = error.code) !== null && _error$code !== void 0 ? _error$code : ''} - ${(_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'Unknown error'}`;
    }
    if (error.response.status === 401) {
      var _error$response2, _error$response2$data, _error$response2$data2, _error$response$data$;
      return `Unauthorized API Error${(_error$response2 = error.response) !== null && _error$response2 !== void 0 && (_error$response2$data = _error$response2.data) !== null && _error$response2$data !== void 0 && (_error$response2$data2 = _error$response2$data.error) !== null && _error$response2$data2 !== void 0 && _error$response2$data2.message ? ` - ${(_error$response$data$ = error.response.data.error) === null || _error$response$data$ === void 0 ? void 0 : _error$response$data$.message}` : ''}`;
    }
    return `API Error: ${(_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : _error$response3.statusText}${(_error$response4 = error.response) !== null && _error$response4 !== void 0 && (_error$response4$data = _error$response4.data) !== null && _error$response4$data !== void 0 && (_error$response4$data2 = _error$response4$data.error) !== null && _error$response4$data2 !== void 0 && _error$response4$data2.message ? ` - ${(_error$response$data$2 = error.response.data.error) === null || _error$response$data$2 === void 0 ? void 0 : _error$response$data$2.message}` : ''}`;
  }
  /**
   * responsible for making a POST request to the external API endpoint and returning the response data
   * @param body The stringified request body to be sent in the POST request.
   */
  async runApi({
    body
  }) {
    const sanitizedBody = (0, _utils.sanitizeRequest)(this.provider, this.url, body, ...('defaultModel' in this.config ? [this.config.defaultModel] : []));
    const axiosOptions = (0, _utils.getAxiosOptions)(this.provider, this.key, false);
    const response = await this.request({
      url: this.url,
      method: 'post',
      responseSchema: _schema.RunActionResponseSchema,
      data: sanitizedBody,
      // give up to 2 minutes for response
      timeout: 120000,
      ...axiosOptions
    });
    return response.data;
  }

  /**
   *  responsible for making a POST request to a specified URL with a given request body.
   *  The method can handle both regular API requests and streaming requests based on the stream parameter.
   *  It uses helper functions getRequestWithStreamOption and getAxiosOptions to prepare the request body and headers respectively.
   *  The response is then processed based on whether it is a streaming response or a regular response.
   * @param body request body for the API request
   * @param stream flag indicating whether it is a streaming request or not
   */
  async streamApi({
    body,
    stream
  }) {
    const executeBody = (0, _utils.getRequestWithStreamOption)(this.provider, this.url, body, stream, ...('defaultModel' in this.config ? [this.config.defaultModel] : []));
    const axiosOptions = (0, _utils.getAxiosOptions)(this.provider, this.key, stream);
    const response = await this.request({
      url: this.url,
      method: 'post',
      responseSchema: stream ? _schema.StreamingResponseSchema : _schema.RunActionResponseSchema,
      data: executeBody,
      ...axiosOptions
    });
    return stream ? (0, _utils.pipeStreamingResponse)(response) : response.data;
  }

  /**
   *  retrieves a dashboard from the Kibana server and checks if the
   *  user has the necessary privileges to access it.
   * @param dashboardId The ID of the dashboard to retrieve.
   */
  async getDashboard({
    dashboardId
  }) {
    const privilege = await this.esClient.transport.request({
      path: '/_security/user/_has_privileges',
      method: 'POST',
      body: {
        index: [{
          names: ['.kibana-event-log-*'],
          allow_restricted_indices: true,
          privileges: ['read']
        }]
      }
    });
    if (!(privilege !== null && privilege !== void 0 && privilege.has_all_requested)) {
      return {
        available: false
      };
    }
    const response = await (0, _create_gen_ai_dashboard.initDashboard)({
      logger: this.logger,
      savedObjectsClient: this.savedObjectsClient,
      dashboardId,
      genAIProvider: 'OpenAI'
    });
    return {
      available: response.success
    };
  }

  /**
   * Responsible for invoking the streamApi method with the provided body and
   * stream parameters set to true. It then returns a Transform stream that processes
   * the response from the streamApi method and returns the response string alone.
   * @param body - the OpenAI Invoke request body
   */
  async invokeStream(body) {
    const res = await this.streamApi({
      body: JSON.stringify(body),
      stream: true
    });
    return res.pipe(new _stream.PassThrough());
  }

  /**
   * Deprecated. Use invokeStream instead.
   * TODO: remove once streaming work is implemented in langchain mode for security solution
   * tracked here: https://github.com/elastic/security-team/issues/7363
   */
  async invokeAI(body) {
    var _res$choices$0$messag;
    const res = await this.runApi({
      body: JSON.stringify(body)
    });
    if (res.choices && res.choices.length > 0 && (_res$choices$0$messag = res.choices[0].message) !== null && _res$choices$0$messag !== void 0 && _res$choices$0$messag.content) {
      const result = res.choices[0].message.content.trim();
      return {
        message: result,
        usage: res.usage
      };
    }
    return {
      message: 'An error occurred sending your message. \n\nAPI Error: The response from OpenAI was in an unrecognized format.',
      ...(res.usage ? {
        usage: res.usage
      } : {
        usage: {
          prompt_tokens: 0,
          completion_tokens: 0,
          total_tokens: 0
        }
      })
    };
  }
}
exports.OpenAIConnector = OpenAIConnector;