"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BedrockConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _aws = _interopRequireDefault(require("aws4"));
var _stream = require("stream");
var _create_gen_ai_dashboard = require("../lib/gen_ai/create_gen_ai_dashboard");
var _schema = require("../../../common/bedrock/schema");
var _constants = require("../../../common/bedrock/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class BedrockConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "url", void 0);
    (0, _defineProperty2.default)(this, "model", void 0);
    this.url = this.config.apiUrl;
    this.model = this.config.defaultModel;
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.SUB_ACTION.RUN,
      method: 'runApi',
      schema: _schema.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.DASHBOARD,
      method: 'getDashboard',
      schema: _schema.DashboardActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.TEST,
      method: 'runApi',
      schema: _schema.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.INVOKE_AI,
      method: 'invokeAI',
      schema: _schema.InvokeAIActionParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.INVOKE_STREAM,
      method: 'invokeStream',
      schema: _schema.InvokeAIActionParamsSchema
    });
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response2, _error$response2$data, _error$response4, _error$response5, _error$response5$data;
    if (!((_error$response = error.response) !== null && _error$response !== void 0 && _error$response.status)) {
      var _error$code, _error$message;
      return `Unexpected API Error: ${(_error$code = error.code) !== null && _error$code !== void 0 ? _error$code : ''} - ${(_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'Unknown error'}`;
    }
    if (error.response.status === 400 && ((_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : (_error$response2$data = _error$response2.data) === null || _error$response2$data === void 0 ? void 0 : _error$response2$data.message) === 'The requested operation is not recognized by the service.') {
      // Leave space in the string below, \n is not being rendered in the UI
      return `API Error: ${error.response.data.message}

The Kibana Connector in use may need to be reconfigured with an updated Amazon Bedrock endpoint, like \`bedrock-runtime\`.`;
    }
    if (error.response.status === 401) {
      var _error$response3, _error$response3$data;
      return `Unauthorized API Error${(_error$response3 = error.response) !== null && _error$response3 !== void 0 && (_error$response3$data = _error$response3.data) !== null && _error$response3$data !== void 0 && _error$response3$data.message ? `: ${error.response.data.message}` : ''}`;
    }
    return `API Error: ${(_error$response4 = error.response) === null || _error$response4 === void 0 ? void 0 : _error$response4.statusText}${(_error$response5 = error.response) !== null && _error$response5 !== void 0 && (_error$response5$data = _error$response5.data) !== null && _error$response5$data !== void 0 && _error$response5$data.message ? ` - ${error.response.data.message}` : ''}`;
  }

  /**
   * provides the AWS signature to the external API endpoint
   * @param body The request body to be signed.
   * @param path The path of the request URL.
   */
  signRequest(body, path, stream) {
    const {
      host
    } = new URL(this.url);
    return _aws.default.sign({
      host,
      headers: stream ? {
        accept: 'application/vnd.amazon.eventstream',
        'Content-Type': 'application/json',
        'x-amzn-bedrock-accept': '*/*'
      } : {
        'Content-Type': 'application/json',
        Accept: '*/*'
      },
      body,
      path,
      // Despite AWS docs, this value does not always get inferred. We need to always send it
      service: 'bedrock'
    }, {
      secretAccessKey: this.secrets.secret,
      accessKeyId: this.secrets.accessKey
    });
  }

  /**
   *  retrieves a dashboard from the Kibana server and checks if the
   *  user has the necessary privileges to access it.
   * @param dashboardId The ID of the dashboard to retrieve.
   */
  async getDashboard({
    dashboardId
  }) {
    const privilege = await this.esClient.transport.request({
      path: '/_security/user/_has_privileges',
      method: 'POST',
      body: {
        index: [{
          names: ['.kibana-event-log-*'],
          allow_restricted_indices: true,
          privileges: ['read']
        }]
      }
    });
    if (!(privilege !== null && privilege !== void 0 && privilege.has_all_requested)) {
      return {
        available: false
      };
    }
    const response = await (0, _create_gen_ai_dashboard.initDashboard)({
      logger: this.logger,
      savedObjectsClient: this.savedObjectsClient,
      dashboardId,
      genAIProvider: 'Bedrock'
    });
    return {
      available: response.success
    };
  }

  /**
   * responsible for making a POST request to the external API endpoint and returning the response data
   * @param body The stringified request body to be sent in the POST request.
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async runApi({
    body,
    model: reqModel
  }) {
    // set model on per request basis
    const path = `/model/${reqModel !== null && reqModel !== void 0 ? reqModel : this.model}/invoke`;
    const signed = this.signRequest(body, path, false);
    const response = await this.request({
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      responseSchema: _schema.RunActionResponseSchema,
      data: body,
      // give up to 2 minutes for response
      timeout: 120000
    });
    return response.data;
  }

  /**
   *  NOT INTENDED TO BE CALLED DIRECTLY
   *  call invokeStream instead
   *  responsible for making a POST request to a specified URL with a given request body.
   *  The response is then processed based on whether it is a streaming response or a regular response.
   * @param body The stringified request body to be sent in the POST request.
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async streamApi({
    body,
    model: reqModel
  }) {
    // set model on per request basis
    const path = `/model/${reqModel !== null && reqModel !== void 0 ? reqModel : this.model}/invoke-with-response-stream`;
    const signed = this.signRequest(body, path, true);
    const response = await this.request({
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      responseSchema: _schema.StreamingResponseSchema,
      data: body,
      responseType: 'stream'
    });
    return response.data.pipe(new _stream.PassThrough());
  }

  /**
   *  takes in an array of messages and a model as inputs. It calls the streamApi method to make a
   *  request to the Bedrock API with the formatted messages and model. It then returns a Transform stream
   *  that pipes the response from the API through the transformToString function,
   *  which parses the proprietary response into a string of the response text alone
   * @param messages An array of messages to be sent to the API
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async invokeStream({
    messages,
    model
  }) {
    const res = await this.streamApi({
      body: JSON.stringify(formatBedrockBody({
        messages
      })),
      model
    });
    return res;
  }

  /**
   * Deprecated. Use invokeStream instead.
   * TODO: remove once streaming work is implemented in langchain mode for security solution
   * tracked here: https://github.com/elastic/security-team/issues/7363
   */
  async invokeAI({
    messages,
    model
  }) {
    const res = await this.runApi({
      body: JSON.stringify(formatBedrockBody({
        messages
      })),
      model
    });
    return {
      message: res.completion.trim()
    };
  }
}
exports.BedrockConnector = BedrockConnector;
const formatBedrockBody = ({
  messages
}) => {
  const combinedMessages = messages.reduce((acc, message) => {
    const {
      role,
      content
    } = message;
    // Bedrock only has Assistant and Human, so 'system' and 'user' will be converted to Human
    const bedrockRole = role === 'assistant' ? '\n\nAssistant:' : '\n\nHuman:';
    return `${acc}${bedrockRole}${content}`;
  }, '');
  return {
    // end prompt in "Assistant:" to avoid the model starting its message with "Assistant:"
    prompt: `${combinedMessages} \n\nAssistant:`,
    max_tokens_to_sample: _constants.DEFAULT_TOKEN_LIMIT,
    temperature: 0.5,
    // prevent model from talking to itself
    stop_sequences: ['\n\nHuman:']
  };
};