"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executor = executor;
exports.getChecksum = getChecksum;
exports.getInvalidComparatorError = getInvalidComparatorError;
exports.getValidTimefieldSort = getValidTimefieldSort;
exports.tryToParseAsDate = tryToParseAsDate;
var _jsSha = require("js-sha256");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/triggers-actions-ui-plugin/common");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common2 = require("../../../common");
var _action_context = require("./action_context");
var _constants = require("./constants");
var _fetch_es_query = require("./lib/fetch_es_query");
var _fetch_search_source_query = require("./lib/fetch_search_source_query");
var _util = require("./util");
var _fetch_esql_query = require("./lib/fetch_esql_query");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function executor(core, options) {
  var _core$http$basePath$p;
  const searchSourceRule = (0, _util.isSearchSourceRule)(options.params.searchType);
  const esqlQueryRule = (0, _util.isEsqlQueryRule)(options.params.searchType);
  const {
    rule: {
      id: ruleId,
      name
    },
    services,
    params,
    state,
    spaceId,
    logger,
    getTimeRange
  } = options;
  const {
    alertsClient,
    scopedClusterClient,
    searchSourceClient,
    share,
    dataViews
  } = services;
  const currentTimestamp = new Date().toISOString();
  const publicBaseUrl = (_core$http$basePath$p = core.http.basePath.publicBaseUrl) !== null && _core$http$basePath$p !== void 0 ? _core$http$basePath$p : '';
  const spacePrefix = spaceId !== 'default' ? `/s/${spaceId}` : '';
  const alertLimit = alertsClient === null || alertsClient === void 0 ? void 0 : alertsClient.getAlertLimitValue();
  const compareFn = _common2.ComparatorFns.get(params.thresholdComparator);
  if (compareFn == null) {
    throw new Error(getInvalidComparatorError(params.thresholdComparator));
  }
  const isGroupAgg = (0, _common.isGroupAggregation)(params.termField);
  // For ungrouped queries, we run the configured query during each rule run, get a hit count
  // and retrieve up to params.size hits. We evaluate the threshold condition using the
  // value of the hit count. If the threshold condition is met, the hits are counted
  // toward the query match and we update the rule state with the timestamp of the latest hit.
  // In the next run of the rule, the latestTimestamp will be used to gate the query in order to
  // avoid counting a document multiple times.
  // latestTimestamp will be ignored if set for grouped queries
  let latestTimestamp = tryToParseAsDate(state.latestTimestamp);
  const {
    dateStart,
    dateEnd
  } = getTimeRange(`${params.timeWindowSize}${params.timeWindowUnit}`);
  const {
    parsedResults,
    link,
    index
  } = searchSourceRule ? await (0, _fetch_search_source_query.fetchSearchSourceQuery)({
    ruleId,
    alertLimit,
    params: params,
    latestTimestamp,
    spacePrefix,
    services: {
      share,
      searchSourceClient,
      logger,
      dataViews
    },
    dateStart,
    dateEnd
  }) : esqlQueryRule ? await (0, _fetch_esql_query.fetchEsqlQuery)({
    ruleId,
    alertLimit,
    params: params,
    spacePrefix,
    publicBaseUrl,
    services: {
      share,
      scopedClusterClient,
      logger
    },
    dateStart,
    dateEnd
  }) : await (0, _fetch_es_query.fetchEsQuery)({
    ruleId,
    name,
    alertLimit,
    params: params,
    timestamp: latestTimestamp,
    publicBaseUrl,
    spacePrefix,
    services: {
      scopedClusterClient,
      logger
    },
    dateStart,
    dateEnd
  });
  const unmetGroupValues = {};
  for (const result of parsedResults.results) {
    var _result$value, _params$threshold;
    const alertId = result.group;
    const value = (_result$value = result.value) !== null && _result$value !== void 0 ? _result$value : result.count;

    // group aggregations use the bucket selector agg to compare conditions
    // within the ES query, so only 'met' results are returned, therefore we don't need
    // to use the compareFn
    const met = isGroupAgg ? true : compareFn(value, params.threshold);
    if (!met) {
      unmetGroupValues[alertId] = value;
      continue;
    }
    const baseContext = {
      title: name,
      date: currentTimestamp,
      value,
      hits: result.hits,
      link
    };
    const baseActiveContext = {
      ...baseContext,
      conditions: (0, _action_context.getContextConditionsDescription)({
        searchType: params.searchType,
        comparator: params.thresholdComparator,
        threshold: params.threshold,
        aggType: params.aggType,
        aggField: params.aggField,
        ...(isGroupAgg ? {
          group: alertId
        } : {})
      })
    };
    const actionContext = (0, _action_context.addMessages)({
      ruleName: name,
      baseContext: baseActiveContext,
      params,
      ...(isGroupAgg ? {
        group: alertId
      } : {}),
      index
    });
    const id = alertId === _common.UngroupedGroupId && !isGroupAgg ? _constants.ConditionMetAlertInstanceId : alertId;
    alertsClient.report({
      id,
      actionGroup: _constants.ActionGroupId,
      state: {
        latestTimestamp,
        dateStart,
        dateEnd
      },
      context: actionContext,
      payload: {
        [_ruleDataUtils.ALERT_URL]: actionContext.link,
        [_ruleDataUtils.ALERT_REASON]: actionContext.message,
        [_.ALERT_TITLE]: actionContext.title,
        [_.ALERT_EVALUATION_CONDITIONS]: actionContext.conditions,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: `${actionContext.value}`,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: ((_params$threshold = params.threshold) === null || _params$threshold === void 0 ? void 0 : _params$threshold.length) === 1 ? params.threshold[0] : null
      }
    });
    if (!isGroupAgg) {
      var _result$hits$find;
      // update the timestamp based on the current search results
      const firstValidTimefieldSort = getValidTimefieldSort((_result$hits$find = result.hits.find(hit => getValidTimefieldSort(hit.sort))) === null || _result$hits$find === void 0 ? void 0 : _result$hits$find.sort);
      if (firstValidTimefieldSort) {
        latestTimestamp = firstValidTimefieldSort;
      }
    }
  }
  alertsClient.setAlertLimitReached(parsedResults.truncated);
  const {
    getRecoveredAlerts
  } = alertsClient;
  for (const recoveredAlert of getRecoveredAlerts()) {
    var _unmetGroupValues$ale, _params$threshold2;
    const alertId = recoveredAlert.alert.getId();
    const baseRecoveryContext = {
      title: name,
      date: currentTimestamp,
      value: (_unmetGroupValues$ale = unmetGroupValues[alertId]) !== null && _unmetGroupValues$ale !== void 0 ? _unmetGroupValues$ale : 0,
      hits: [],
      link,
      conditions: (0, _action_context.getContextConditionsDescription)({
        searchType: params.searchType,
        comparator: params.thresholdComparator,
        threshold: params.threshold,
        isRecovered: true,
        aggType: params.aggType,
        aggField: params.aggField,
        ...(isGroupAgg ? {
          group: alertId
        } : {})
      })
    };
    const recoveryContext = (0, _action_context.addMessages)({
      ruleName: name,
      baseContext: baseRecoveryContext,
      params,
      isRecovered: true,
      ...(isGroupAgg ? {
        group: alertId
      } : {}),
      index
    });
    alertsClient === null || alertsClient === void 0 ? void 0 : alertsClient.setAlertData({
      id: alertId,
      context: recoveryContext,
      payload: {
        [_ruleDataUtils.ALERT_URL]: recoveryContext.link,
        [_ruleDataUtils.ALERT_REASON]: recoveryContext.message,
        [_.ALERT_TITLE]: recoveryContext.title,
        [_.ALERT_EVALUATION_CONDITIONS]: recoveryContext.conditions,
        [_ruleDataUtils.ALERT_EVALUATION_VALUE]: `${recoveryContext.value}`,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: ((_params$threshold2 = params.threshold) === null || _params$threshold2 === void 0 ? void 0 : _params$threshold2.length) === 1 ? params.threshold[0] : null
      }
    });
  }
  return {
    state: {
      latestTimestamp
    }
  };
}
function getValidTimefieldSort(sortValues = []) {
  for (const sortValue of sortValues) {
    const sortDate = tryToParseAsDate(sortValue);
    if (sortDate) {
      return sortDate;
    }
  }
}
function tryToParseAsDate(sortValue) {
  const sortDate = typeof sortValue === 'string' ? Date.parse(sortValue) : sortValue;
  if (sortDate && !isNaN(sortDate)) {
    return new Date(sortDate).toISOString();
  }
}
function getChecksum(params) {
  return _jsSha.sha256.create().update(JSON.stringify(params));
}
function getInvalidComparatorError(comparator) {
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.invalidComparatorErrorMessage', {
    defaultMessage: 'invalid thresholdComparator specified: {comparator}',
    values: {
      comparator
    }
  });
}