"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SelectableSpacesControl = void 0;
require("./selectable_spaces_control.scss");
var _eui = require("@elastic/eui");
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _no_spaces_available = require("./no_spaces_available");
var _common = require("../../../common");
var _constants = require("../../../common/constants");
var _space_avatar = require("../../space_avatar");
var _spaces_context = require("../../spaces_context");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// No need to wrap LazySpaceAvatar in an error boundary, because it is one of the first chunks loaded when opening Kibana.
const LazySpaceAvatar = /*#__PURE__*/(0, _react.lazy)(() => (0, _space_avatar.getSpaceAvatarComponent)().then(component => ({
  default: component
})));
const ROW_HEIGHT = 40;
const APPEND_ACTIVE_SPACE = /*#__PURE__*/_react.default.createElement(_eui.EuiBadge, {
  color: "hollow"
}, _i18n.i18n.translate('xpack.spaces.shareToSpace.currentSpaceBadge', {
  defaultMessage: 'This space'
}));
const APPEND_CANNOT_SELECT = /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
  content: _i18n.i18n.translate('xpack.spaces.shareToSpace.partiallyAuthorizedSpaceTooltip.unchecked', {
    defaultMessage: 'You need additional privileges to select this space.'
  }),
  position: "left",
  type: "iInCircle"
});
const APPEND_CANNOT_DESELECT = /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
  content: _i18n.i18n.translate('xpack.spaces.shareToSpace.partiallyAuthorizedSpaceTooltip.checked', {
    defaultMessage: 'You need additional privileges to deselect this space.'
  }),
  position: "left",
  type: "iInCircle"
});
const APPEND_PROHIBITED = /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
  title: _i18n.i18n.translate('xpack.spaces.shareToSpace.prohibitedSpaceTooltipTitle', {
    defaultMessage: 'Cannot share to this space'
  }),
  content: _i18n.i18n.translate('xpack.spaces.shareToSpace.prohibitedSpaceTooltip', {
    defaultMessage: 'A copy of this saved object or a related object exists in this space.'
  }),
  position: "left",
  type: "iInCircle"
});
const APPEND_FEATURE_IS_DISABLED = /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
  content: _i18n.i18n.translate('xpack.spaces.shareToSpace.featureIsDisabledTooltip', {
    defaultMessage: 'This feature is disabled in this space.'
  }),
  position: "left",
  type: "warning",
  color: "warning"
});
const SelectableSpacesControl = props => {
  const {
    spaces,
    shareOptions,
    onChange,
    enableCreateNewSpaceLink,
    enableSpaceAgnosticBehavior,
    prohibitedSpaces
  } = props;
  const {
    services
  } = (0, _spaces_context.useSpaces)();
  const {
    application,
    docLinks
  } = services;
  const {
    selectedSpaceIds,
    initiallySelectedSpaceIds
  } = shareOptions;
  const activeSpaceId = !enableSpaceAgnosticBehavior && spaces.find(space => space.isActiveSpace).id;
  const isGlobalControlChecked = selectedSpaceIds.includes(_constants.ALL_SPACES_ID);
  const filteredSpaces = spaces.filter(
  // filter out spaces that are not already selected and have the feature disabled in that space
  ({
    id,
    isFeatureDisabled
  }) => !isFeatureDisabled || initiallySelectedSpaceIds.includes(id) || isGlobalControlChecked);
  const options = filteredSpaces.sort(createSpacesComparator(activeSpaceId)).map(space => {
    const checked = selectedSpaceIds.includes(space.id);
    const {
      isAvatarDisabled,
      ...additionalProps
    } = getAdditionalProps(space, activeSpaceId, checked, isGlobalControlChecked, prohibitedSpaces);
    return {
      label: space.name,
      prepend: /*#__PURE__*/_react.default.createElement(LazySpaceAvatar, {
        space: space,
        isDisabled: isAvatarDisabled,
        size: 's'
      }),
      // wrapped in a Suspense below
      checked: checked || isGlobalControlChecked ? 'on' : undefined,
      ['data-space-id']: space.id,
      ['data-test-subj']: `sts-space-selector-row-${space.id}`,
      ...(isGlobalControlChecked && {
        disabled: true
      }),
      ...additionalProps
    };
  });
  function updateSelectedSpaces(spaceOptions) {
    const selectedOptions = spaceOptions.filter(x => x.checked && x['data-space-id'] !== activeSpaceId).map(x => x['data-space-id']);
    const updatedSpaceIds = [...selectedOptions, ...selectedSpaceIds.filter(x => x === _constants.UNKNOWN_SPACE) // preserve any unknown spaces (to keep the "selected spaces" count accurate)
    ];

    onChange(updatedSpaceIds);
  }
  const getUnknownSpacesLabel = () => {
    const count = selectedSpaceIds.filter(id => id === _constants.UNKNOWN_SPACE).length;
    if (!count) {
      return null;
    }
    const hiddenCount = selectedSpaceIds.filter(id => id === _constants.UNKNOWN_SPACE).length;
    const docLink = docLinks === null || docLinks === void 0 ? void 0 : docLinks.links.security.kibanaPrivileges;
    return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      size: "s",
      color: "subdued"
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.spaces.shareToSpace.unknownSpacesLabel.text",
      defaultMessage: "To view {hiddenCount} hidden spaces, you need {additionalPrivilegesLink}.",
      values: {
        hiddenCount,
        additionalPrivilegesLink: /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
          href: docLink,
          target: "_blank"
        }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
          id: "xpack.spaces.shareToSpace.unknownSpacesLabel.additionalPrivilegesLink",
          defaultMessage: "additional privileges"
        }))
      }
    })));
  };
  const getNoSpacesAvailable = () => {
    if (enableCreateNewSpaceLink && spaces.length < 2) {
      return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
        grow: false
      }, /*#__PURE__*/_react.default.createElement(_no_spaces_available.NoSpacesAvailable, {
        application: application
      }));
    }
    return null;
  };

  // if space-agnostic behavior is not enabled, the active space is not selected or deselected by the user, so we have to artificially pad the count for this label
  const selectedCountPad = enableSpaceAgnosticBehavior ? 0 : 1;
  const selectedCount = isGlobalControlChecked ? filteredSpaces.length : selectedSpaceIds.filter(id => id !== _constants.ALL_SPACES_ID && id !== _constants.UNKNOWN_SPACE).length + selectedCountPad;
  const selectSpacesLabel = _i18n.i18n.translate('xpack.spaces.shareToSpace.shareModeControl.selectSpacesLabel', {
    defaultMessage: 'Select spaces'
  });
  const selectedSpacesLabel = _i18n.i18n.translate('xpack.spaces.shareToSpace.shareModeControl.selectedCountLabel', {
    defaultMessage: '{selectedCount}/{totalCount} selected',
    values: {
      selectedCount,
      totalCount: filteredSpaces.length
    }
  });
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFormRow, {
    label: selectSpacesLabel,
    labelAppend: /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
      size: "xs"
    }, selectedSpacesLabel),
    fullWidth: true
  }, /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null)), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    gutterSize: "none",
    responsive: false,
    style: {
      minHeight: 200
    }
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_react.Suspense, {
    fallback: /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, null)
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSelectable, {
    options: options,
    onChange: newOptions => updateSelectedSpaces(newOptions),
    listProps: {
      bordered: true,
      rowHeight: ROW_HEIGHT,
      className: 'spcShareToSpace__spacesList',
      'data-test-subj': 'sts-form-space-selector'
    },
    height: "full",
    searchable: options.length > _common.SPACE_SEARCH_COUNT_THRESHOLD
  }, (list, search) => {
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, search, list);
  }))), getUnknownSpacesLabel(), getNoSpacesAvailable()));
};

/**
 * Gets additional props for the selection option.
 */
exports.SelectableSpacesControl = SelectableSpacesControl;
function getAdditionalProps(space, activeSpaceId, checked, isGlobalControlChecked, prohibitedSpaces) {
  if (space.id === activeSpaceId) {
    return {
      append: APPEND_ACTIVE_SPACE,
      disabled: true,
      checked: 'on'
    };
  }
  if (!isGlobalControlChecked && !space.isAuthorizedForPurpose('shareSavedObjectsIntoSpace')) {
    return {
      append: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, checked ? APPEND_CANNOT_DESELECT : APPEND_CANNOT_SELECT, space.isFeatureDisabled ? APPEND_FEATURE_IS_DISABLED : null),
      ...(space.isFeatureDisabled && {
        isAvatarDisabled: true
      }),
      disabled: true
    };
  }
  if (prohibitedSpaces.has(space.id) || prohibitedSpaces.has(_constants.ALL_SPACES_ID)) {
    return {
      append: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, APPEND_PROHIBITED, space.isFeatureDisabled ? APPEND_FEATURE_IS_DISABLED : null),
      ...(space.isFeatureDisabled && {
        isAvatarDisabled: true
      }),
      disabled: true
    };
  }
  if (space.isFeatureDisabled) {
    return {
      append: APPEND_FEATURE_IS_DISABLED,
      isAvatarDisabled: true
    };
  }
  return {};
}

/**
 * Given the active space, create a comparator to sort a SpacesDataEntry array so that the active space is at the beginning, and space(s) for
 * which the current feature is disabled are all at the end.
 */
function createSpacesComparator(activeSpaceId) {
  return (a, b) => {
    if (a.id === activeSpaceId) {
      return -1;
    }
    if (b.id === activeSpaceId) {
      return 1;
    }
    if (a.isFeatureDisabled !== b.isFeatureDisabled) {
      return a.isFeatureDisabled ? 1 : -1;
    }
    return 0;
  };
}