"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateRepository = exports.INVALID_NAME_CHARS = void 0;
var _text = require("../text");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INVALID_NAME_CHARS = ['"', '*', '\\', '<', '|', ',', '>', '/', '?', '#'];
exports.INVALID_NAME_CHARS = INVALID_NAME_CHARS;
const isStringEmpty = str => {
  return str ? !Boolean(str.trim()) : true;
};
const doesStringContainChar = (string, char) => {
  const chars = Array.isArray(char) ? char : [char];
  const total = chars.length;
  let containsChar = false;
  let charFound = null;
  for (let i = 0; i < total; i++) {
    if (string.includes(chars[i])) {
      containsChar = true;
      charFound = chars[i];
      break;
    }
  }
  return {
    containsChar,
    charFound
  };
};
const validateRepository = (repository, validateSettings = true) => {
  const {
    name,
    type,
    settings
  } = repository;
  const {
    i18n
  } = _text.textService;
  const validation = {
    isValid: true,
    errors: {}
  };
  if (validateSettings) {
    const settingsValidation = validateRepositorySettings(type, settings);
    if (Object.keys(settingsValidation).length) {
      validation.errors.settings = settingsValidation;
    }
  }
  if (isStringEmpty(name)) {
    validation.errors.name = [i18n.translate('xpack.snapshotRestore.repositoryValidation.nameRequired', {
      defaultMessage: 'Repository name is required.'
    })];
  }
  if (name.includes(' ')) {
    validation.errors.name = [i18n.translate('xpack.snapshotRestore.repositoryValidation.nameValidation.errorSpace', {
      defaultMessage: 'Spaces are not allowed in the name.'
    })];
  }
  const nameCharValidation = doesStringContainChar(name, INVALID_NAME_CHARS);
  if (nameCharValidation.containsChar) {
    validation.errors.name = [i18n.translate('xpack.snapshotRestore.repositoryValidation.nameValidation.invalidCharacter', {
      defaultMessage: 'Character "{char}" is not allowed in the name.',
      values: {
        char: nameCharValidation.charFound
      }
    })];
  }
  if (isStringEmpty(type) || type === _constants.REPOSITORY_TYPES.source && isStringEmpty(settings.delegateType)) {
    validation.errors.type = [i18n.translate('xpack.snapshotRestore.repositoryValidation.delegateTypeRequired', {
      defaultMessage: 'Type is required.'
    })];
  }
  if (Object.keys(validation.errors).length) {
    validation.isValid = false;
  }
  return validation;
};
exports.validateRepository = validateRepository;
const validateRepositorySettings = (type, settings) => {
  switch (type) {
    case _constants.REPOSITORY_TYPES.fs:
      return validateFSRepositorySettings(settings);
    case _constants.REPOSITORY_TYPES.url:
      return validateReadonlyRepositorySettings(settings);
    case _constants.REPOSITORY_TYPES.source:
      return validateRepositorySettings(settings.delegateType, settings);
    case _constants.REPOSITORY_TYPES.s3:
      return validateS3RepositorySettings(settings);
    case _constants.REPOSITORY_TYPES.gcs:
      return validateGCSRepositorySettings(settings);
    case _constants.REPOSITORY_TYPES.hdfs:
      return validateHDFSRepositorySettings(settings);
    // No validation on settings needed for azure (all settings are optional)
    default:
      return {};
  }
};
const validateFSRepositorySettings = settings => {
  const i18n = _text.textService.i18n;
  const validation = {};
  const {
    location
  } = settings;
  if (isStringEmpty(location)) {
    validation.location = [i18n.translate('xpack.snapshotRestore.repositoryValidation.locationRequired', {
      defaultMessage: 'Location is required.'
    })];
  }
  return validation;
};
const validateReadonlyRepositorySettings = settings => {
  const i18n = _text.textService.i18n;
  const validation = {};
  const {
    url
  } = settings;
  if (isStringEmpty(url)) {
    validation.url = [i18n.translate('xpack.snapshotRestore.repositoryValidation.urlRequired', {
      defaultMessage: 'URL is required.'
    })];
  }
  return validation;
};
const validateS3RepositorySettings = settings => {
  const i18n = _text.textService.i18n;
  const validation = {};
  const {
    bucket
  } = settings;
  if (isStringEmpty(bucket)) {
    validation.bucket = [i18n.translate('xpack.snapshotRestore.repositoryValidation.bucketRequired', {
      defaultMessage: 'Bucket is required.'
    })];
  }
  return validation;
};
const validateGCSRepositorySettings = settings => {
  const i18n = _text.textService.i18n;
  const validation = {};
  const {
    bucket
  } = settings;
  if (isStringEmpty(bucket)) {
    validation.bucket = [i18n.translate('xpack.snapshotRestore.repositoryValidation.bucketRequired', {
      defaultMessage: 'Bucket is required.'
    })];
  }
  return validation;
};
const validateHDFSRepositorySettings = settings => {
  const i18n = _text.textService.i18n;
  const validation = {};
  const {
    uri,
    path
  } = settings;
  if (isStringEmpty(uri)) {
    validation.uri = [i18n.translate('xpack.snapshotRestore.repositoryValidation.uriRequired', {
      defaultMessage: 'URI is required.'
    })];
  }
  if (isStringEmpty(path)) {
    validation.path = [i18n.translate('xpack.snapshotRestore.repositoryValidation.pathRequired', {
      defaultMessage: 'Path is required.'
    })];
  }
  return validation;
};