"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useTogglePanel = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _common = require("@kbn/security-solution-plugin/common");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _product = require("../../../common/product");
var _storage = require("../storage");
var _reducer = require("../reducer");
var _types = require("../types");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const syncExpandedCardStepsToStorageFromURL = maybeStepId => {
  const {
    matchedCard,
    matchedStep
  } = (0, _helpers.findCardSectionByStepId)(maybeStepId);
  const hasStepContent = matchedStep && matchedStep.description;
  if (matchedCard && matchedStep && hasStepContent) {
    _storage.getStartedStorage.resetAllExpandedCardStepsToStorage();
    _storage.getStartedStorage.addExpandedCardStepToStorage(matchedCard.id, matchedStep.id);
  }
};
const syncExpandedCardStepsFromStorageToURL = (expandedCardSteps, callback) => {
  const expandedCardStep = Object.values(expandedCardSteps).find(expandedCard => expandedCard.expandedSteps.length > 0);
  if (expandedCardStep !== null && expandedCardStep !== void 0 && expandedCardStep.expandedSteps[0]) {
    const {
      matchedCard,
      matchedStep
    } = (0, _helpers.findCardSectionByStepId)(expandedCardStep === null || expandedCardStep === void 0 ? void 0 : expandedCardStep.expandedSteps[0]);
    callback === null || callback === void 0 ? void 0 : callback({
      matchedCard,
      matchedStep
    });
  }
};
const useTogglePanel = ({
  productTypes
}) => {
  const {
    navigateTo
  } = (0, _securitySolutionNavigation.useNavigateTo)();
  const {
    hash: detailName
  } = (0, _reactRouterDom.useLocation)();
  const stepIdFromHash = detailName.split('#')[1];
  const {
    getAllFinishedStepsFromStorage,
    getActiveProductsFromStorage,
    toggleActiveProductsInStorage,
    addExpandedCardStepToStorage,
    addFinishedStepToStorage,
    removeFinishedStepFromStorage,
    removeExpandedCardStepFromStorage,
    resetAllExpandedCardStepsToStorage,
    getAllExpandedCardStepsFromStorage
  } = _storage.getStartedStorage;
  const finishedStepsInitialStates = (0, _react.useMemo)(() => (0, _reducer.getFinishedStepsInitialStates)({
    finishedSteps: getAllFinishedStepsFromStorage()
  }), [getAllFinishedStepsFromStorage]);
  const activeProductsInitialStates = (0, _react.useMemo)(() => {
    var _Set;
    const activeProductsFromStorage = (0, _reducer.getActiveProductsInitialStates)({
      activeProducts: getActiveProductsFromStorage()
    });
    return activeProductsFromStorage.size > 0 ? activeProductsFromStorage : (_Set = new Set(productTypes.map(({
      product_line: productLine
    }) => _product.ProductLine[productLine]))) !== null && _Set !== void 0 ? _Set : new Set([_product.ProductLine.security, _product.ProductLine.endpoint, _product.ProductLine.cloud]);
  }, [getActiveProductsFromStorage, productTypes]);
  const {
    activeSections: activeSectionsInitialStates,
    totalActiveSteps: totalActiveStepsInitialStates,
    totalStepsLeft: totalStepsLeftInitialStates
  } = (0, _react.useMemo)(() => (0, _reducer.getActiveSectionsInitialStates)({
    activeProducts: activeProductsInitialStates,
    finishedSteps: finishedStepsInitialStates
  }), [activeProductsInitialStates, finishedStepsInitialStates]);
  const expandedCardsInitialStates = (0, _react.useMemo)(() => {
    if (stepIdFromHash) {
      syncExpandedCardStepsToStorageFromURL(stepIdFromHash);
    }
    return getAllExpandedCardStepsFromStorage();
  }, [getAllExpandedCardStepsFromStorage, stepIdFromHash]);
  const onStepClicked = (0, _react.useCallback)(({
    stepId,
    cardId,
    isExpanded
  }) => {
    dispatch({
      type: _types.GetStartedPageActions.ToggleExpandedStep,
      payload: {
        stepId,
        cardId,
        isStepExpanded: isExpanded
      }
    });
    if (isExpanded) {
      // It allows Only One step open at a time
      resetAllExpandedCardStepsToStorage();
      addExpandedCardStepToStorage(cardId, stepId);
    } else {
      removeExpandedCardStepFromStorage(cardId, stepId);
    }
  }, [addExpandedCardStepToStorage, removeExpandedCardStepFromStorage, resetAllExpandedCardStepsToStorage]);
  const [state, dispatch] = (0, _react.useReducer)(_reducer.reducer, {
    activeProducts: activeProductsInitialStates,
    activeSections: activeSectionsInitialStates,
    expandedCardSteps: expandedCardsInitialStates,
    finishedSteps: finishedStepsInitialStates,
    totalActiveSteps: totalActiveStepsInitialStates,
    totalStepsLeft: totalStepsLeftInitialStates
  });
  const toggleTaskCompleteStatus = (0, _react.useCallback)(({
    stepId,
    cardId,
    sectionId,
    undo
  }) => {
    dispatch({
      type: undo ? _types.GetStartedPageActions.RemoveFinishedStep : _types.GetStartedPageActions.AddFinishedStep,
      payload: {
        stepId,
        cardId,
        sectionId
      }
    });
    if (undo) {
      removeFinishedStepFromStorage(cardId, stepId);
    } else {
      addFinishedStepToStorage(cardId, stepId);
    }
  }, [addFinishedStepToStorage, removeFinishedStepFromStorage]);
  const onProductSwitchChanged = (0, _react.useCallback)(section => {
    dispatch({
      type: _types.GetStartedPageActions.ToggleProduct,
      payload: {
        section: section.id
      }
    });
    toggleActiveProductsInStorage(section.id);
  }, [toggleActiveProductsInStorage]);
  (0, _react.useEffect)(() => {
    /** Handle landing on the page without hash
     ** e.g.: https://localhost:5601/app/security/get_started
     ** If there is no expanded card step in storage, do nothing.
     ** If there is expanded card step in storage, sync it to the url.
     **/
    if (!stepIdFromHash) {
      // If all steps are collapsed, do nothing
      if (Object.values(state.expandedCardSteps).every(c => !c.isExpanded)) {
        return;
      }
      syncExpandedCardStepsFromStorageToURL(expandedCardsInitialStates, ({
        matchedStep
      }) => {
        if (!matchedStep) return;
        navigateTo({
          deepLinkId: _common.SecurityPageName.landing,
          path: `#${matchedStep.id}`
        });
      });
    }
  }, [expandedCardsInitialStates, getAllExpandedCardStepsFromStorage, navigateTo, state.expandedCardSteps, stepIdFromHash]);
  (0, _react.useEffect)(() => {
    /** Handle hash change and expand the target step.
     ** e.g.: https://localhost:5601/app/security/get_started#create_your_first_project
     **/
    if (stepIdFromHash) {
      const {
        matchedCard,
        matchedStep,
        matchedSection
      } = (0, _helpers.findCardSectionByStepId)(stepIdFromHash);
      const hasStepContent = matchedStep && matchedStep.description;
      if (hasStepContent && matchedCard && matchedStep && matchedSection) {
        var _state$expandedCardSt;
        // If the step is already expanded, do nothing
        if ((_state$expandedCardSt = state.expandedCardSteps[matchedCard.id]) !== null && _state$expandedCardSt !== void 0 && _state$expandedCardSt.expandedSteps.includes(matchedStep.id)) {
          return;
        }
        // Toggle step and sync the expanded card step to storage & reducer
        onStepClicked({
          stepId: matchedStep.id,
          cardId: matchedCard.id,
          sectionId: matchedSection.id,
          isExpanded: true
        });
        navigateTo({
          deepLinkId: _common.SecurityPageName.landing,
          path: `#${matchedStep.id}`
        });
      }
    }
  }, [navigateTo, onStepClicked, state.expandedCardSteps, stepIdFromHash]);
  return {
    state,
    onStepClicked,
    toggleTaskCompleteStatus,
    onProductSwitchChanged
  };
};
exports.useTogglePanel = useTogglePanel;