"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryPrebuiltRuleAlertsTaskConfig = createTelemetryPrebuiltRuleAlertsTaskConfig;
var _constants = require("../constants");
var _helpers = require("../helpers");
var _filterlists = require("../filterlists");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryPrebuiltRuleAlertsTaskConfig(maxTelemetryBatch) {
  const taskVersion = '1.2.0';
  return {
    type: 'security:telemetry-prebuilt-rule-alerts',
    title: 'Security Solution - Prebuilt Rule and Elastic ML Alerts Telemetry',
    interval: '1h',
    timeout: '15m',
    version: taskVersion,
    runTask: async (taskId, logger, receiver, sender, taskExecutionPeriod) => {
      const startTime = Date.now();
      const taskName = 'Security Solution - Prebuilt Rule and Elastic ML Alerts Telemetry';
      try {
        const [clusterInfoPromise, licenseInfoPromise, packageVersion] = await Promise.allSettled([receiver.fetchClusterInfo(), receiver.fetchLicenseInfo(), receiver.fetchDetectionRulesPackageVersion()]);
        const clusterInfo = clusterInfoPromise.status === 'fulfilled' ? clusterInfoPromise.value : {};
        const licenseInfo = licenseInfoPromise.status === 'fulfilled' ? licenseInfoPromise.value : {};
        const packageInfo = packageVersion.status === 'fulfilled' ? packageVersion.value : undefined;
        const index = receiver.getAlertsIndex();
        if (index === undefined) {
          (0, _helpers.tlog)(logger, `alerts index is not ready yet, skipping telemetry task`);
          return 0;
        }
        let fetchMore = true;
        let searchAfterValue;
        let pitId = await receiver.openPointInTime(index);
        while (fetchMore) {
          const {
            moreToFetch,
            newPitId,
            searchAfter,
            alerts
          } = await receiver.fetchPrebuiltRuleAlertsBatch(pitId, searchAfterValue);
          if (alerts.length === 0) {
            return 0;
          }
          fetchMore = moreToFetch;
          searchAfterValue = searchAfter;
          pitId = newPitId;
          const processedAlerts = alerts.map(event => (0, _filterlists.copyAllowlistedFields)(_filterlists.filterList.prebuiltRulesAlerts, event));
          const sanitizedAlerts = processedAlerts.map(event => (0, _helpers.processK8sUsernames)(clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_uuid, event));
          const enrichedAlerts = sanitizedAlerts.map(event => ({
            ...event,
            licence_id: licenseInfo === null || licenseInfo === void 0 ? void 0 : licenseInfo.uid,
            cluster_uuid: clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_uuid,
            cluster_name: clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_name,
            package_version: packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.version,
            task_version: taskVersion
          }));
          (0, _helpers.tlog)(logger, `sending ${enrichedAlerts.length} elastic prebuilt alerts`);
          const batches = (0, _helpers.batchTelemetryRecords)(enrichedAlerts, maxTelemetryBatch);
          const promises = batches.map(async batch => {
            sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_DETECTION_ALERTS, batch);
          });
          await Promise.all(promises);
        }
        await receiver.closePointInTime(pitId);
        return 0;
      } catch (err) {
        logger.error('could not complete prebuilt alerts telemetry task');
        await sender.sendOnDemand(_constants.TASK_METRICS_CHANNEL, [(0, _helpers.createTaskMetric)(taskName, false, startTime, err.message)]);
        return 0;
      }
    }
  };
}