"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateRulesStream = exports.validateRules = exports.sortImports = exports.migrateLegacyInvestigationFields = exports.createRulesAndExceptionsStreamFromNdJson = void 0;
var _fp = require("lodash/fp");
var _utils = require("@kbn/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _rule_management = require("../../../../../../common/api/detection_engine/rule_management");
var _create_stream_from_ndjson = require("../../../../../utils/read_stream/create_stream_from_ndjson");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Validates exception lists and items schemas
 */
const validateRulesStream = () => {
  return (0, _utils.createMapStream)(items => ({
    actionConnectors: items.actionConnectors,
    exceptions: items.exceptions,
    rules: validateRules(items.rules)
  }));
};
exports.validateRulesStream = validateRulesStream;
const validateRules = rules => {
  return rules.map(obj => {
    if (obj instanceof Error) {
      return obj;
    }
    const result = _rule_management.RuleToImport.safeParse(obj);
    if (!result.success) {
      return new _securitysolutionEsUtils.BadRequestError((0, _zodHelpers.stringifyZodError)(result.error));
    }
    const validationErrors = (0, _rule_management.validateRuleToImport)(result.data);
    if (validationErrors.length) {
      return new _securitysolutionEsUtils.BadRequestError(validationErrors.join());
    }
    return result.data;
  });
};

/**
 * Sorts the exceptions into the lists and items.
 * We do this because we don't want the order of the exceptions
 * in the import to matter. If we didn't sort, then some items
 * might error if the list has not yet been created
 */
exports.validateRules = validateRules;
const sortImports = () => {
  return (0, _utils.createReduceStream)((acc, importItem) => {
    if ((0, _fp.has)('list_id', importItem) || (0, _fp.has)('item_id', importItem) || (0, _fp.has)('entries', importItem)) {
      return {
        ...acc,
        exceptions: [...acc.exceptions, importItem]
      };
    }
    if ((0, _fp.has)('attributes', importItem)) {
      return {
        ...acc,
        actionConnectors: [...acc.actionConnectors, importItem]
      };
    } else {
      return {
        ...acc,
        rules: [...acc.rules, importItem]
      };
    }
  }, {
    exceptions: [],
    rules: [],
    actionConnectors: []
  });
};
exports.sortImports = sortImports;
const migrateLegacyInvestigationFields = () => {
  return (0, _utils.createMapStream)(obj => {
    if (obj != null && 'investigation_fields' in obj && Array.isArray(obj.investigation_fields)) {
      if (obj.investigation_fields.length) {
        return {
          ...obj,
          investigation_fields: {
            field_names: obj.investigation_fields
          }
        };
      } else {
        const {
          investigation_fields: _,
          ...rest
        } = obj;
        return rest;
      }
    }
    return obj;
  });
};

// TODO: Capture both the line number and the rule_id if you have that information for the error message
// eventually and then pass it down so we can give error messages on the line number
exports.migrateLegacyInvestigationFields = migrateLegacyInvestigationFields;
const createRulesAndExceptionsStreamFromNdJson = ruleLimit => {
  return [(0, _utils.createSplitStream)('\n'), (0, _create_stream_from_ndjson.parseNdjsonStrings)(), (0, _create_stream_from_ndjson.filterExportedCounts)(), migrateLegacyInvestigationFields(), sortImports(), validateRulesStream(), (0, _create_stream_from_ndjson.createRulesLimitStream)(ruleLimit), (0, _utils.createConcatStream)([])];
};
exports.createRulesAndExceptionsStreamFromNdJson = createRulesAndExceptionsStreamFromNdJson;