"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRuleActionsWithMigratedResults = exports.swapNonDefaultSpaceIdWithDestinationId = exports.returnErroredImportResult = exports.mapSOErrorToRuleError = exports.mapActionIdToNewDestinationId = exports.handleActionsHaveNoConnectors = exports.handleActionConnectorsErrors = exports.getActionConnectorRules = exports.filterExistingActionConnectors = exports.checkIfActionsHaveMissingConnectors = void 0;
var _lodash = require("lodash");
var _utils = require("../../../../../routes/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const returnErroredImportResult = error => ({
  success: false,
  errors: [handleActionConnectorsErrors(error)],
  successCount: 0,
  warnings: []
});
exports.returnErroredImportResult = returnErroredImportResult;
const handleActionsHaveNoConnectors = (actionsIds, actionConnectorRules) => {
  const ruleIds = [...new Set(Object.values(actionConnectorRules).flat())].join();
  if (actionsIds && actionsIds.length) {
    const errors = [];
    const errorMessage = actionsIds.length > 1 ? 'connectors are missing. Connector ids missing are:' : 'connector is missing. Connector id missing is:';
    errors.push((0, _utils.createBulkErrorObject)({
      id: actionsIds.join(),
      statusCode: 404,
      message: `${actionsIds.length} ${errorMessage} ${actionsIds.join(', ')}`,
      ruleId: ruleIds
    }));
    return {
      success: false,
      errors,
      successCount: 0,
      warnings: []
    };
  }
  return {
    success: true,
    errors: [],
    successCount: 0,
    warnings: []
  };
};
exports.handleActionsHaveNoConnectors = handleActionsHaveNoConnectors;
const handleActionConnectorsErrors = (error, id) => {
  let statusCode = null;
  let message = '';
  if ('output' in error) {
    var _output$payload;
    statusCode = error.output.statusCode;
    message = (_output$payload = error.output.payload) === null || _output$payload === void 0 ? void 0 : _output$payload.message;
  }
  switch (statusCode) {
    case null:
      return (0, _utils.createBulkErrorObject)({
        statusCode: 500,
        message: (error === null || error === void 0 ? void 0 : error.type) === 'conflict' ? 'There is a conflict' : error.message ? error.message : ''
      });
    case 403:
      return (0, _utils.createBulkErrorObject)({
        id,
        statusCode,
        message: `You may not have actions privileges required to import rules with actions: ${message}`
      });
    default:
      return (0, _utils.createBulkErrorObject)({
        id,
        statusCode,
        message
      });
  }
};
exports.handleActionConnectorsErrors = handleActionConnectorsErrors;
const mapSOErrorToRuleError = errors => {
  return errors.map(({
    id,
    error
  }) => handleActionConnectorsErrors(error, id));
};
exports.mapSOErrorToRuleError = mapSOErrorToRuleError;
const filterExistingActionConnectors = async (actionsClient, actionsIds) => {
  const storedConnectors = await actionsClient.getAll();
  const storedActionIds = storedConnectors.map(({
    id
  }) => id);
  return actionsIds.filter(id => !storedActionIds.includes(id));
};
exports.filterExistingActionConnectors = filterExistingActionConnectors;
const getActionConnectorRules = rules => rules.reduce((acc, rule) => {
  var _rule$actions;
  if (rule instanceof Error) return acc;
  (_rule$actions = rule.actions) === null || _rule$actions === void 0 ? void 0 : _rule$actions.forEach(({
    id
  }) => acc[id] = [...(acc[id] || []), rule.rule_id]);
  return acc;
}, {});
exports.getActionConnectorRules = getActionConnectorRules;
const checkIfActionsHaveMissingConnectors = (actionConnectors, newIdsToAdd, actionConnectorRules) => {
  // if new action-connectors don't have exported connectors will fail with missing connectors
  if (actionConnectors.length < newIdsToAdd.length) {
    const actionConnectorsIds = actionConnectors.map(({
      id
    }) => id);
    const missingActionConnector = newIdsToAdd.filter(id => !actionConnectorsIds.includes(id));
    const missingActionRules = (0, _lodash.pick)(actionConnectorRules, [...missingActionConnector]);
    return handleActionsHaveNoConnectors(missingActionConnector, missingActionRules);
  }
  return null;
};
exports.checkIfActionsHaveMissingConnectors = checkIfActionsHaveMissingConnectors;
const mapActionIdToNewDestinationId = connectorsImportResult => {
  return connectorsImportResult.reduce((acc, {
    destinationId,
    id
  }) => {
    acc[id] = destinationId || id;
    return acc;
  }, {});
};
exports.mapActionIdToNewDestinationId = mapActionIdToNewDestinationId;
const swapNonDefaultSpaceIdWithDestinationId = (rule, actionIdDestinationIdLookup) => {
  var _rule$actions2;
  return (_rule$actions2 = rule.actions) === null || _rule$actions2 === void 0 ? void 0 : _rule$actions2.map(action => {
    const destinationId = actionIdDestinationIdLookup[action.id];
    return {
      ...action,
      id: destinationId
    };
  });
};
/*
// When a connector is exported from one namespace and imported to another, it does not result in an error, but instead a new object is created with
// new destination id and id will have the old  origin id, so in order to be able to use the newly generated Connectors id, this util is used to swap the old id with the
// new destination Id
*/
exports.swapNonDefaultSpaceIdWithDestinationId = swapNonDefaultSpaceIdWithDestinationId;
const updateRuleActionsWithMigratedResults = (rules, connectorsImportResult) => {
  const actionIdDestinationIdLookup = mapActionIdToNewDestinationId(connectorsImportResult);
  return rules.map(rule => {
    if (rule instanceof Error) return rule;
    return {
      ...rule,
      actions: swapNonDefaultSpaceIdWithDestinationId(rule, actionIdDestinationIdLookup)
    };
  });
};
exports.updateRuleActionsWithMigratedResults = updateRuleActionsWithMigratedResults;