"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.writeActionToIndices = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../../../../common/endpoint/service/response_actions/constants");
var _constants2 = require("../../../../../common/endpoint/constants");
var _utils = require("../../../utils");
var _action_errors = require("./action_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const writeActionToIndices = async ({
  actionID,
  agents,
  esClient,
  endpointContext,
  minimumLicenseRequired,
  payload
}) => {
  var _payload$comment, _getActionParameters;
  const logger = endpointContext.logFactory.get('createResponseAction');
  const licenseService = endpointContext.service.getLicenseService();
  const doc = {
    '@timestamp': (0, _moment.default)().toISOString(),
    agent: {
      id: payload.endpoint_ids
    },
    EndpointActions: {
      action_id: actionID,
      expiration: (0, _moment.default)().add(2, 'weeks').toISOString(),
      type: 'INPUT_ACTION',
      input_type: 'endpoint',
      data: {
        command: payload.command,
        comment: (_payload$comment = payload.comment) !== null && _payload$comment !== void 0 ? _payload$comment : undefined,
        ...(payload.alert_ids ? {
          alert_id: payload.alert_ids
        } : {}),
        ...(payload.hosts ? {
          hosts: payload.hosts
        } : {}),
        parameters: (_getActionParameters = getActionParameters(payload)) !== null && _getActionParameters !== void 0 ? _getActionParameters : undefined
      }
    },
    user: {
      id: payload.user ? payload.user.username : 'unknown'
    },
    ...(0, _action_errors.addErrorsToActionIfAny)({
      agents,
      licenseService,
      minimumLicenseRequired
    }),
    ...addRuleInfoToAction(payload)
  };

  // if .logs-endpoint.actions data stream exists
  // try to create action request record in .logs-endpoint.actions DS as the current user
  // (from >= v7.16, use this check to ensure the current user has privileges to write to the new index)
  // and allow only users with superuser privileges to write to fleet indices
  const doesLogsEndpointActionsDsExist = await (0, _utils.doLogsEndpointActionDsExists)({
    esClient,
    logger,
    dataStreamName: _constants2.ENDPOINT_ACTIONS_DS
  });

  // if the new endpoint indices/data streams exists
  // write the action request to the new endpoint index
  if (doesLogsEndpointActionsDsExist) {
    const logsEndpointActionsResult = await esClient.index({
      index: _constants2.ENDPOINT_ACTIONS_INDEX,
      document: {
        ...doc,
        agent: {
          id: payload.endpoint_ids
        }
      },
      refresh: 'wait_for'
    }, {
      meta: true
    });
    if (logsEndpointActionsResult.statusCode !== 201) {
      throw new Error(logsEndpointActionsResult.body.result);
    }
  }
  if (!doc.error) {
    // add signature to doc
    const fleetActionDoc = {
      ...doc.EndpointActions,
      '@timestamp': doc['@timestamp'],
      agents,
      timeout: 300,
      // 5 minutes
      user_id: doc.user.id
    };
    const fleetActionDocSignature = await endpointContext.service.getMessageSigningService().sign(fleetActionDoc);
    const signedFleetActionDoc = {
      ...fleetActionDoc,
      signed: {
        data: fleetActionDocSignature.data.toString('base64'),
        signature: fleetActionDocSignature.signature
      }
    };
    // write actions to .fleet-actions index
    try {
      const fleetActionsClient = await endpointContext.service.getFleetActionsClient();
      await fleetActionsClient.create(signedFleetActionDoc);
    } catch (e) {
      // create entry in .logs-endpoint.action.responses-default data stream
      // when writing to .fleet-actions fails
      if (doesLogsEndpointActionsDsExist) {
        await createFailedActionResponseEntry({
          esClient,
          doc: {
            '@timestamp': (0, _moment.default)().toISOString(),
            agent: doc.agent,
            EndpointActions: {
              action_id: doc.EndpointActions.action_id,
              completed_at: (0, _moment.default)().toISOString(),
              started_at: (0, _moment.default)().toISOString(),
              data: doc.EndpointActions.data
            }
          },
          logger
        });
      }
      throw e;
    }
  }
};
exports.writeActionToIndices = writeActionToIndices;
const createFailedActionResponseEntry = async ({
  esClient,
  doc,
  logger
}) => {
  try {
    await esClient.index({
      index: `${_constants2.ENDPOINT_ACTION_RESPONSES_DS}-default`,
      document: {
        ...doc,
        error: {
          code: _constants2.failedFleetActionErrorCode,
          message: 'Failed to deliver action request to fleet'
        }
      }
    });
  } catch (error) {
    logger.error((0, _utils.wrapErrorIfNeeded)(error));
  }
};
const addRuleInfoToAction = payload => {
  if (payload.rule_id && payload.rule_name) {
    return {
      rule: {
        id: payload.rule_id,
        name: payload.rule_name
      }
    };
  }
};
const getActionParameters = action => {
  var _action$parameters;
  // set timeout to 4h (if not specified or when timeout is specified as 0) when command is `execute`
  if (action.command === 'execute') {
    const actionRequestParams = action.parameters;
    if (typeof (actionRequestParams === null || actionRequestParams === void 0 ? void 0 : actionRequestParams.timeout) === 'undefined') {
      return {
        ...actionRequestParams,
        timeout: _constants.DEFAULT_EXECUTE_ACTION_TIMEOUT
      };
    }
    return actionRequestParams;
  }

  // for all other commands return the parameters as is
  return (_action$parameters = action.parameters) !== null && _action$parameters !== void 0 ? _action$parameters : undefined;
};