"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointActionsClient = void 0;
var _update_cases = require("../create/update_cases");
var _base_actions_provider = require("../../../lib/response_actions/base_actions_provider");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EndpointActionsClient extends _base_actions_provider.BaseResponseActionsClient {
  async checkAgentIds(ids) {
    const foundEndpointHosts = await this.options.endpointContext.service.getEndpointMetadataService().getMetadataForEndpoints(this.options.esClient, [...new Set(ids)]);
    const validIds = foundEndpointHosts.map(endpoint => endpoint.elastic.agent.id);
    const invalidIds = ids.filter(id => !validIds.includes(id));
    if (invalidIds.length) {
      this.log.debug(`The following agent ids are not valid: ${JSON.stringify(invalidIds)}`);
    }
    return {
      valid: validIds,
      invalid: invalidIds,
      allValid: invalidIds.length === 0,
      hosts: foundEndpointHosts
    };
  }
  async handleResponseAction(command, options) {
    const agentIds = await this.checkAgentIds(options.endpoint_ids);
    const createPayload = {
      ...options,
      command,
      user: {
        username: this.options.username
      }
    };
    const response = await this.options.endpointContext.service.getActionCreateService().createAction(createPayload, agentIds.valid);
    await this.updateCases(createPayload, agentIds.hosts);
    return response;
  }
  async updateCases(createActionPayload, endpointData) {
    return (0, _update_cases.updateCases)({
      casesClient: this.options.casesClient,
      createActionPayload,
      endpointData
    });
  }
  async isolate(options) {
    return this.handleResponseAction('isolate', options);
  }
  async release(options) {
    return this.handleResponseAction('unisolate', options);
  }
  async killProcess(options) {
    return this.handleResponseAction('kill-process', options);
  }
  async suspendProcess(options) {
    return this.handleResponseAction('suspend-process', options);
  }
  async runningProcesses(options) {
    return this.handleResponseAction('running-processes', options);
  }
  async getFile(options) {
    return this.handleResponseAction('get-file', options);
  }
  async execute(options) {
    return this.handleResponseAction('execute', options);
  }
  async upload(options) {
    const fleetFiles = await this.options.endpointContext.service.getFleetToHostFilesClient();
    const fileStream = options.file;
    const {
      file: _,
      parameters: userParams,
      ...actionPayload
    } = options;
    const uploadParameters = {
      ...userParams,
      file_id: '',
      file_name: '',
      file_sha256: '',
      file_size: 0
    };
    const createdFile = await fleetFiles.create(fileStream, actionPayload.endpoint_ids);
    uploadParameters.file_id = createdFile.id;
    uploadParameters.file_name = createdFile.name;
    uploadParameters.file_sha256 = createdFile.sha256;
    uploadParameters.file_size = createdFile.size;
    const createFileActionOptions = {
      ...actionPayload,
      parameters: uploadParameters,
      command: 'upload'
    };
    try {
      const createdAction = await this.handleResponseAction('upload', createFileActionOptions);

      // Update the file meta to include the action id, and if any errors (unlikely),
      // then just log them and still allow api to return success since the action has
      // already been created and potentially dispatched to Endpoint. Action ID is not
      // needed by the Endpoint or fleet-server's API, so no need to fail here
      try {
        await fleetFiles.update(uploadParameters.file_id, {
          actionId: createdAction.id
        });
      } catch (e) {
        this.log.warn(`Attempt to update File meta with Action ID failed: ${e.message}`, e);
      }
      return createdAction;
    } catch (err) {
      if (uploadParameters.file_id) {
        // Try to delete the created file since creating the action threw an error
        try {
          await fleetFiles.delete(uploadParameters.file_id);
        } catch (e) {
          this.log.error(`Attempt to clean up file id [${uploadParameters.file_id}] (after action creation was unsuccessful) failed; ${e.message}`, e);
        }
      }
      throw err;
    }
  }
}
exports.EndpointActionsClient = EndpointActionsClient;