"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.temporarilyDisableInteractiveChildTabIndexes = exports.draggableKeyDownHandler = void 0;
var _securitysolutionTGrid = require("@kbn/securitysolution-t-grid");
var _public = require("@kbn/timelines-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Temporarily disables tab focus on child links of the draggable to work
 * around an issue where tab focus becomes stuck on the interactive children
 *
 * NOTE: This function is (intentionally) only effective when used in a key
 * event handler, because it automatically restores focus capabilities on
 * the next tick.
 */
const temporarilyDisableInteractiveChildTabIndexes = draggableElement => {
  const interactiveChildren = draggableElement.querySelectorAll('a, button');
  interactiveChildren.forEach(interactiveChild => {
    interactiveChild.setAttribute('tabindex', '-1'); // DOM mutation
  });

  // restore the default tabindexs on the next tick:
  setTimeout(() => {
    interactiveChildren.forEach(interactiveChild => {
      interactiveChild.setAttribute('tabindex', '0'); // DOM mutation
    });
  }, 0);
};
exports.temporarilyDisableInteractiveChildTabIndexes = temporarilyDisableInteractiveChildTabIndexes;
const draggableKeyDownHandler = ({
  beginDrag,
  cancelDragActions,
  closePopover,
  draggableElement,
  dragActions,
  dragToLocation,
  endDrag,
  keyboardEvent,
  openPopover,
  setDragActions
}) => {
  let currentPosition = null;
  switch (keyboardEvent.key) {
    case ' ':
      if (!dragActions) {
        // start dragging, because space was pressed
        if (closePopover != null) {
          closePopover();
        }
        setDragActions(beginDrag());
      } else {
        // end dragging, because space was pressed
        endDrag(dragActions);
        setDragActions(null);
      }
      break;
    case 'Escape':
      cancelDragActions();
      break;
    case 'Tab':
      // IMPORTANT: we do NOT want to stop propagation and prevent default when Tab is pressed
      temporarilyDisableInteractiveChildTabIndexes(draggableElement);
      break;
    case 'ArrowUp':
      currentPosition = draggableElement.getBoundingClientRect();
      dragToLocation({
        dragActions,
        position: {
          x: currentPosition.x,
          y: currentPosition.y - _securitysolutionTGrid.KEYBOARD_DRAG_OFFSET
        }
      });
      break;
    case 'ArrowDown':
      currentPosition = draggableElement.getBoundingClientRect();
      dragToLocation({
        dragActions,
        position: {
          x: currentPosition.x,
          y: currentPosition.y + _securitysolutionTGrid.KEYBOARD_DRAG_OFFSET
        }
      });
      break;
    case 'ArrowLeft':
      currentPosition = draggableElement.getBoundingClientRect();
      dragToLocation({
        dragActions,
        position: {
          x: currentPosition.x - _securitysolutionTGrid.KEYBOARD_DRAG_OFFSET,
          y: currentPosition.y
        }
      });
      break;
    case 'ArrowRight':
      currentPosition = draggableElement.getBoundingClientRect();
      dragToLocation({
        dragActions,
        position: {
          x: currentPosition.x + _securitysolutionTGrid.KEYBOARD_DRAG_OFFSET,
          y: currentPosition.y
        }
      });
      break;
    case 'Enter':
      (0, _public.stopPropagationAndPreventDefault)(keyboardEvent); // prevents the first item in the popover from getting an errant ENTER
      if (!dragActions && openPopover != null) {
        openPopover();
      }
      break;
    default:
      break;
  }
};
exports.draggableKeyDownHandler = draggableKeyDownHandler;