"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useStream = void 0;
var _react = require("react");
var _stream_observable = require("./stream_observable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A hook that takes a ReadableStreamDefaultReader and returns an object with properties and functions
 * that can be used to handle streaming data from a readable stream
 * @param amendMessage - handles the amended message
 * @param content - the content of the message. If provided, the function will not use the reader to stream data.
 * @param reader - The readable stream reader used to stream data. If provided, the function will use this reader to stream data.
 * @param isError - indicates whether the reader response is an error message or not
 */
const useStream = ({
  amendMessage,
  content,
  connectorTypeTitle,
  reader,
  isError
}) => {
  const [pendingMessage, setPendingMessage] = (0, _react.useState)();
  const [loading, setLoading] = (0, _react.useState)(false);
  const [error, setError] = (0, _react.useState)();
  const [subscription, setSubscription] = (0, _react.useState)();
  const observer$ = (0, _react.useMemo)(() => content == null && reader != null ? (0, _stream_observable.getStreamObservable)({
    connectorTypeTitle,
    reader,
    setLoading,
    isError
  }) : (0, _stream_observable.getPlaceholderObservable)(), [content, isError, reader, connectorTypeTitle]);
  const onCompleteStream = (0, _react.useCallback)(() => {
    subscription === null || subscription === void 0 ? void 0 : subscription.unsubscribe();
    setLoading(false);
    amendMessage(pendingMessage !== null && pendingMessage !== void 0 ? pendingMessage : '');
  }, [amendMessage, pendingMessage, subscription]);
  const [complete, setComplete] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (complete) {
      setComplete(false);
      onCompleteStream();
    }
  }, [complete, onCompleteStream]);
  (0, _react.useEffect)(() => {
    const newSubscription = observer$.subscribe({
      next: ({
        message,
        loading: isLoading
      }) => {
        setLoading(isLoading);
        setPendingMessage(message);
      },
      complete: () => {
        setComplete(true);
      },
      error: err => {
        setError(err.message);
      }
    });
    setSubscription(newSubscription);
  }, [observer$]);
  return {
    error,
    isLoading: loading,
    isStreaming: loading && pendingMessage != null,
    pendingMessage: pendingMessage !== null && pendingMessage !== void 0 ? pendingMessage : '',
    setComplete
  };
};
exports.useStream = useStream;