"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexFleetEndpointPolicy = exports.deleteIndexedFleetEndpointPolicies = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _lodash = require("lodash");
var _format_axios_error = require("../format_axios_error");
var _usage_tracker = require("./usage_tracker");
var _package = require("../utils/package");
var _policy_config = require("../models/policy_config");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
var TimeoutsInMS;
/**
 * Create an endpoint Integration Policy (and associated Agent Policy) via Fleet
 * (NOTE: ensure that fleet is setup first before calling this loading function)
 */
(function (TimeoutsInMS) {
  TimeoutsInMS[TimeoutsInMS["TEN_SECONDS"] = 10000] = "TEN_SECONDS";
  TimeoutsInMS[TimeoutsInMS["FIVE_MINUTES"] = 300000] = "FIVE_MINUTES";
})(TimeoutsInMS || (TimeoutsInMS = {}));
const indexFleetEndpointPolicy = _usage_tracker.usageTracker.track('indexFleetEndpointPolicy', async (kbnClient, policyName, endpointPackageVersion, agentPolicyName, log) => {
  const response = {
    integrationPolicies: [],
    agentPolicies: []
  };
  const packageVersion = endpointPackageVersion !== null && endpointPackageVersion !== void 0 ? endpointPackageVersion : await getDefaultEndpointPackageVersion(kbnClient);

  // Create Agent Policy first
  const newAgentPolicyData = {
    name: agentPolicyName || `Policy for ${policyName} (${Math.random().toString(36).substr(2, 5)})`,
    description: `Policy created with endpoint data generator (${policyName})`,
    namespace: 'default',
    monitoring_enabled: ['logs', 'metrics']
  };
  let agentPolicy;
  try {
    agentPolicy = await kbnClient.request({
      path: _common.AGENT_POLICY_API_ROUTES.CREATE_PATTERN,
      headers: {
        'elastic-api-version': _common.API_VERSIONS.public.v1
      },
      method: 'POST',
      body: newAgentPolicyData
    }).catch(_utils.wrapErrorAndRejectPromise);
  } catch (error) {
    throw new Error(`create fleet agent policy failed ${error}`);
  }
  response.agentPolicies.push(agentPolicy.data.item);

  // Create integration (package) policy
  const newPackagePolicyData = {
    name: policyName,
    // skip_ensure_installed: true,
    description: 'Protect the worlds data',
    policy_id: agentPolicy.data.item.id,
    enabled: true,
    inputs: [{
      type: 'endpoint',
      enabled: true,
      streams: [],
      config: {
        policy: {
          value: (0, _policy_config.policyFactory)()
        }
      }
    }],
    namespace: 'default',
    package: {
      name: 'endpoint',
      title: 'Elastic Defend',
      version: packageVersion
    }
  };
  const createPackagePolicy = async () => kbnClient.request({
    path: _common.PACKAGE_POLICY_API_ROUTES.CREATE_PATTERN,
    method: 'POST',
    body: newPackagePolicyData,
    headers: {
      'elastic-api-version': _common.API_VERSIONS.public.v1
    }
  }).catch(_format_axios_error.catchAxiosErrorFormatAndThrow).then(res => res.data);
  const started = new Date();
  const hasTimedOut = () => {
    const elapsedTime = Date.now() - started.getTime();
    return elapsedTime > TimeoutsInMS.FIVE_MINUTES;
  };
  let packagePolicy;
  log === null || log === void 0 ? void 0 : log.debug(`Creating integration policy with name: ${policyName}`);
  while (!packagePolicy && !hasTimedOut()) {
    packagePolicy = await (0, _utils.retryOnError)(async () => createPackagePolicy(), [..._utils.RETRYABLE_TRANSIENT_ERRORS, 'resource_not_found_exception'], log);
    if (!packagePolicy) {
      await new Promise(resolve => setTimeout(resolve, TimeoutsInMS.TEN_SECONDS));
    }
  }
  if (!packagePolicy) {
    throw new Error(`Create package policy failed`);
  }
  log === null || log === void 0 ? void 0 : log.verbose(`Integration policy created:`, JSON.stringify(packagePolicy, null, 2));
  response.integrationPolicies.push(packagePolicy.item);
  return response;
});
exports.indexFleetEndpointPolicy = indexFleetEndpointPolicy;
/**
 * Delete indexed Fleet Endpoint integration policies along with their respective Agent Policies.
 * Prior to calling this function, ensure that no agents are associated with the Agent Policy.
 * (NOTE: ensure that fleet is setup first before calling this loading function)
 * @param kbnClient
 * @param indexData
 */
const deleteIndexedFleetEndpointPolicies = async (kbnClient, indexData) => {
  const response = {
    integrationPolicies: undefined,
    agentPolicies: undefined
  };
  if (indexData.integrationPolicies.length) {
    response.integrationPolicies = (await kbnClient.request({
      path: _common.PACKAGE_POLICY_API_ROUTES.DELETE_PATTERN,
      headers: {
        'elastic-api-version': _common.API_VERSIONS.public.v1
      },
      method: 'POST',
      body: {
        packagePolicyIds: indexData.integrationPolicies.map(policy => policy.id)
      }
    }).catch(_utils.wrapErrorAndRejectPromise)).data;
  }
  if (indexData.agentPolicies.length) {
    response.agentPolicies = [];
    for (const agentPolicy of indexData.agentPolicies) {
      response.agentPolicies.push((await kbnClient.request({
        path: _common.AGENT_POLICY_API_ROUTES.DELETE_PATTERN,
        headers: {
          'elastic-api-version': _common.API_VERSIONS.public.v1
        },
        method: 'POST',
        body: {
          agentPolicyId: agentPolicy.id
        }
      }).catch(_utils.wrapErrorAndRejectPromise)).data);
    }
  }
  return response;
};
exports.deleteIndexedFleetEndpointPolicies = deleteIndexedFleetEndpointPolicies;
const getDefaultEndpointPackageVersion = _usage_tracker.usageTracker.track('getDefaultEndpointPackageVersion', (0, _lodash.memoize)(async kbnClient => {
  return (await (0, _package.getEndpointPackageInfo)(kbnClient)).version;
}, kbnClient => {
  return kbnClient.resolveUrl('/');
}));