"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validFeatureIds = exports.isValidFeatureId = exports.getSafeSortIds = exports.getEsQueryConfig = exports.AlertConsumers = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * registering a new instance of the rule data client
 * in a new plugin will require updating the below data structure
 * to include the index name where the alerts as data will be written to.
 */

const AlertConsumers = {
  APM: 'apm',
  LOGS: 'logs',
  INFRASTRUCTURE: 'infrastructure',
  OBSERVABILITY: 'observability',
  SLO: 'slo',
  SIEM: 'siem',
  UPTIME: 'uptime',
  ML: 'ml'
};

// TODO: remove 'in-progress' after migration to 'acknowledged'
exports.AlertConsumers = AlertConsumers;
const validFeatureIds = Object.values(AlertConsumers).map(v => v);
exports.validFeatureIds = validFeatureIds;
const isValidFeatureId = a => typeof a === 'string' && validFeatureIds.includes(a);

/**
 * Prevent javascript from returning Number.MAX_SAFE_INTEGER when Elasticsearch expects
 * Java's Long.MAX_VALUE. This happens when sorting fields by date which are
 * unmapped in the provided index
 *
 * Ref: https://github.com/elastic/elasticsearch/issues/28806#issuecomment-369303620
 *
 * return stringified Long.MAX_VALUE if we receive Number.MAX_SAFE_INTEGER
 * @param sortIds estypes.SortResults | undefined
 * @returns SortResults
 */
exports.isValidFeatureId = isValidFeatureId;
const getSafeSortIds = sortIds => {
  if (sortIds == null) {
    return sortIds;
  }
  return sortIds.map(sortId => {
    // haven't determined when we would receive a null value for a sort id
    // but in case we do, default to sending the stringified Java max_int
    if (sortId == null || sortId === '' || sortId >= Number.MAX_SAFE_INTEGER) {
      return '9223372036854775807';
    }
    return sortId;
  });
};
exports.getSafeSortIds = getSafeSortIds;
const getEsQueryConfig = params => {
  const defaultConfigValues = {
    allowLeadingWildcards: true,
    queryStringOptions: {
      analyze_wildcard: true
    },
    ignoreFilterIfFieldNotInIndex: false,
    dateFormatTZ: 'Zulu'
  };
  if (params == null) {
    return defaultConfigValues;
  }
  const paramKeysWithValues = Object.keys(params).reduce((acc, key) => {
    const configKey = key;
    if (params[configKey] != null) {
      return {
        [key]: params[configKey],
        ...acc
      };
    }
    return {
      [key]: defaultConfigValues[configKey],
      ...acc
    };
  }, {});
  return paramKeysWithValues;
};
exports.getEsQueryConfig = getEsQueryConfig;