"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportingAPIClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _coreHttpCommon = require("@kbn/core-http-common");
var _i18n = require("@kbn/i18n");
var _reportingCommon = require("@kbn/reporting-common");
var _rison = _interopRequireDefault(require("@kbn/rison"));
var _moment = _interopRequireDefault(require("moment"));
var _queryString = require("query-string");
var _constants = require("../../../common/constants");
var _job_completion_notifications = require("../../notifier/job_completion_notifications");
var _job = require("../job");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * For convenience, apps do not have to provide the browserTimezone and Kibana version.
 * Those fields are added in this client as part of the service.
 * TODO: export a type like this to other plugins: https://github.com/elastic/kibana/issues/107085
 */

/**
 * Client class for interacting with Reporting APIs
 * @implements IReportingAPI
 */
class ReportingAPIClient {
  constructor(http, uiSettings, kibanaVersion) {
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "getManagementLink", () => this.http.basePath.prepend(_reportingCommon.REPORTING_MANAGEMENT_HOME));
    (0, _defineProperty2.default)(this, "getDownloadLink", jobId => this.getReportURL(jobId));
    (0, _defineProperty2.default)(this, "getServerBasePath", () => this.http.basePath.serverBasePath);
    this.uiSettings = uiSettings;
    this.kibanaVersion = kibanaVersion;
    this.http = http;
  }
  getKibanaAppHref(job) {
    const searchParams = (0, _queryString.stringify)({
      jobId: job.id
    });
    const path = (0, _reportingCommon.buildKibanaPath)({
      basePath: this.http.basePath.serverBasePath,
      spaceId: job.spaceId,
      appPath: (0, _reportingCommon.getRedirectAppPath)()
    });
    const href = `${path}?${searchParams}`;
    return href;
  }

  /**
   * Get the internal URL
   */
  getReportURL(jobId) {
    const downloadLink = this.http.basePath.prepend(`${_constants.INTERNAL_ROUTES.JOBS.DOWNLOAD_PREFIX}/${jobId}?${_coreHttpCommon.ELASTIC_INTERNAL_ORIGIN_QUERY_PARAM}=true`);
    return downloadLink;
  }
  downloadReport(jobId) {
    const location = this.getReportURL(jobId);
    window.open(location);
  }
  async deleteReport(jobId) {
    return await this.http.delete(`${_constants.INTERNAL_ROUTES.JOBS.DELETE_PREFIX}/${jobId}`);
  }
  async list(page = 0, jobIds = []) {
    const query = {
      page
    };
    if (jobIds.length > 0) {
      // Only getting the first 10, to prevent URL overflows
      query.ids = jobIds.slice(0, 10).join(',');
    }
    const jobQueueEntries = await this.http.get(_constants.INTERNAL_ROUTES.JOBS.LIST, {
      query,
      asSystemRequest: true
    });
    return jobQueueEntries.map(report => new _job.Job(report));
  }
  async total() {
    return await this.http.get(_constants.INTERNAL_ROUTES.JOBS.COUNT, {
      asSystemRequest: true
    });
  }
  async getError(jobId) {
    var _job$warnings;
    const job = await this.getInfo(jobId);
    if ((_job$warnings = job.warnings) !== null && _job$warnings !== void 0 && _job$warnings[0]) {
      // the error message of a failed report is a singular string in the warnings array
      return job.warnings[0];
    }
    return _i18n.i18n.translate('xpack.reporting.apiClient.unknownError', {
      defaultMessage: `Report job {job} failed. Error unknown.`,
      values: {
        job: jobId
      }
    });
  }
  async getInfo(jobId) {
    const report = await this.http.get(`${_constants.INTERNAL_ROUTES.JOBS.INFO_PREFIX}/${jobId}`);
    return new _job.Job(report);
  }
  async findForJobIds(jobIds) {
    const reports = await this.http.fetch(_constants.INTERNAL_ROUTES.JOBS.LIST, {
      query: {
        page: 0,
        ids: jobIds.join(',')
      },
      method: 'GET'
    });
    return reports.map(report => new _job.Job(report));
  }

  /**
   * Returns a string for the public API endpoint used to automate the generation of reports
   * This string must be shown when the user selects the option to view/copy the POST URL
   */
  getReportingPublicJobPath(exportType, jobParams) {
    const params = (0, _queryString.stringify)({
      jobParams: _rison.default.encode(jobParams)
    });
    return `${this.http.basePath.prepend(_constants.PUBLIC_ROUTES.GENERATE_PREFIX)}/${exportType}?${params}`;
  }

  /**
   * Calls the internal API to generate a report job on-demand
   */
  async createReportingJob(exportType, jobParams) {
    const jobParamsRison = _rison.default.encode(jobParams);
    const resp = await this.http.post(`${_constants.INTERNAL_ROUTES.GENERATE_PREFIX}/${exportType}`, {
      method: 'POST',
      body: JSON.stringify({
        jobParams: jobParamsRison
      })
    });
    (0, _job_completion_notifications.add)(resp.job.id);
    return new _job.Job(resp.job);
  }
  async createImmediateReport(baseParams) {
    const {
      objectType: _objectType,
      ...params
    } = baseParams; // objectType is not needed for immediate download api
    return this.http.post(_constants.INTERNAL_ROUTES.DOWNLOAD_CSV, {
      asResponse: true,
      body: JSON.stringify(params)
    });
  }
  getDecoratedJobParams(baseParams) {
    // If the TZ is set to the default "Browser", it will not be useful for
    // server-side export. We need to derive the timezone and pass it as a param
    // to the export API.
    const browserTimezone = this.uiSettings.get('dateFormat:tz') === 'Browser' ? _moment.default.tz.guess() : this.uiSettings.get('dateFormat:tz');
    return {
      browserTimezone,
      version: this.kibanaVersion,
      ...baseParams
    };
  }
  verifyBrowser() {
    return this.http.post(_constants.INTERNAL_ROUTES.DIAGNOSE.BROWSER);
  }
  verifyScreenCapture() {
    return this.http.post(_constants.INTERNAL_ROUTES.DIAGNOSE.SCREENSHOT);
  }
  migrateReportingIndicesIlmPolicy() {
    return this.http.put(_constants.INTERNAL_ROUTES.MIGRATE.MIGRATE_ILM_POLICY);
  }
}
exports.ReportingAPIClient = ReportingAPIClient;