"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchSloList = useFetchSloList;
var _i18n = require("@kbn/i18n");
var _reactQuery = require("@tanstack/react-query");
var _react = require("react");
var _constants = require("../../../common/slo/constants");
var _constants2 = require("../../constants");
var _kibana_react = require("../../utils/kibana_react");
var _query_key_factory = require("./query_key_factory");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useFetchSloList({
  kqlQuery = '',
  page = 1,
  sortBy = 'status',
  sortDirection = 'desc',
  shouldRefetch,
  perPage = _constants.DEFAULT_SLO_PAGE_SIZE
} = {}) {
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana_react.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  const [stateRefetchInterval, setStateRefetchInterval] = (0, _react.useState)(_constants2.SLO_SHORT_REFETCH_INTERVAL);
  const {
    isInitialLoading,
    isLoading,
    isError,
    isSuccess,
    isRefetching,
    data
  } = (0, _reactQuery.useQuery)({
    queryKey: _query_key_factory.sloKeys.list({
      kqlQuery,
      page,
      perPage,
      sortBy,
      sortDirection
    }),
    queryFn: async ({
      signal
    }) => {
      const response = await http.get(`/api/observability/slos`, {
        query: {
          ...(kqlQuery && {
            kqlQuery
          }),
          ...(sortBy && {
            sortBy
          }),
          ...(sortDirection && {
            sortDirection
          }),
          ...(page && {
            page
          }),
          ...(perPage && {
            perPage
          })
        },
        signal
      });
      return response;
    },
    cacheTime: 0,
    refetchOnWindowFocus: false,
    refetchInterval: shouldRefetch ? stateRefetchInterval : undefined,
    retry: (failureCount, error) => {
      if (String(error) === 'Error: Forbidden') {
        return false;
      }
      return failureCount < 4;
    },
    onSuccess: ({
      results
    }) => {
      queryClient.invalidateQueries({
        queryKey: _query_key_factory.sloKeys.historicalSummaries(),
        exact: false
      });
      queryClient.invalidateQueries({
        queryKey: _query_key_factory.sloKeys.activeAlerts(),
        exact: false
      });
      queryClient.invalidateQueries({
        queryKey: _query_key_factory.sloKeys.rules(),
        exact: false
      });
      if (!shouldRefetch) {
        return;
      }
      if (results.find(slo => slo.summary.status === 'NO_DATA' || !slo.summary)) {
        setStateRefetchInterval(_constants2.SLO_SHORT_REFETCH_INTERVAL);
      } else {
        setStateRefetchInterval(_constants2.SLO_LONG_REFETCH_INTERVAL);
      }
    },
    onError: error => {
      toasts.addError(error, {
        title: _i18n.i18n.translate('xpack.observability.slo.list.errorNotification', {
          defaultMessage: 'Something went wrong while fetching SLOs'
        })
      });
    }
  });
  return {
    data,
    isInitialLoading,
    isLoading,
    isRefetching,
    isSuccess,
    isError
  };
}