"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityAIAssistantClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = require("@hapi/boom");
var _gptTokenizer = require("gpt-tokenizer");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _stream = require("stream");
var _uuid = require("uuid");
var _conversation_complete = require("../../../common/conversation_complete");
var _types = require("../../../common/types");
var _concatenate_openai_chunks = require("../../../common/utils/concatenate_openai_chunks");
var _process_openai_stream = require("../../../common/utils/process_openai_stream");
var _knowledge_base_service = require("../knowledge_base_service");
var _get_access_query = require("../util/get_access_query");
var _stream_into_observable = require("../util/stream_into_observable");
var _handle_llm_response = require("./handle_llm_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ObservabilityAIAssistantClient {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "getConversationWithMetaFields", async conversationId => {
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: this.dependencies.resources.aliases.conversations,
        query: {
          bool: {
            filter: [...(0, _get_access_query.getAccessQuery)({
              user: this.dependencies.user,
              namespace: this.dependencies.namespace
            }), {
              term: {
                'conversation.id': conversationId
              }
            }]
          }
        },
        size: 1,
        terminate_after: 1
      });
      return response.hits.hits[0];
    });
    (0, _defineProperty2.default)(this, "getConversationUpdateValues", now => {
      return {
        conversation: {
          last_updated: now
        },
        user: this.dependencies.user,
        namespace: this.dependencies.namespace
      };
    });
    (0, _defineProperty2.default)(this, "get", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      return conversation._source;
    });
    (0, _defineProperty2.default)(this, "delete", async conversationId => {
      const conversation = await this.getConversationWithMetaFields(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      await this.dependencies.esClient.asInternalUser.delete({
        id: conversation._id,
        index: conversation._index,
        refresh: true
      });
    });
    (0, _defineProperty2.default)(this, "complete", async params => {
      const stream = new _stream.PassThrough();
      const {
        messages,
        connectorId,
        signal,
        functionClient,
        persist
      } = params;
      let conversationId = '';
      let title = '';
      if ('conversationId' in params) {
        conversationId = params.conversationId;
      }
      if ('title' in params) {
        title = params.title || '';
      }
      function write(event) {
        if (stream.destroyed) {
          return Promise.resolve();
        }
        return new Promise((resolve, reject) => {
          stream.write(`${JSON.stringify(event)}\n`, 'utf-8', err => {
            if (err) {
              reject(err);
              return;
            }
            resolve();
          });
        });
      }
      function fail(error) {
        const code = (0, _conversation_complete.isChatCompletionError)(error) ? error.code : undefined;
        write({
          type: _conversation_complete.StreamingChatResponseEventType.ConversationCompletionError,
          error: {
            message: error.message,
            stack: error.stack,
            code
          }
        }).finally(() => {
          stream.end();
        });
      }
      let numFunctionsCalled = 0;
      const MAX_FUNCTION_CALLS = 5;
      const MAX_FUNCTION_RESPONSE_TOKEN_COUNT = 4000;
      const next = async nextMessages => {
        var _lastMessage$message$;
        const lastMessage = (0, _lodash.last)(nextMessages);
        const isUserMessage = (lastMessage === null || lastMessage === void 0 ? void 0 : lastMessage.message.role) === _types.MessageRole.User;
        const isUserMessageWithoutFunctionResponse = isUserMessage && !(lastMessage !== null && lastMessage !== void 0 && lastMessage.message.name);
        const recallFirst = isUserMessageWithoutFunctionResponse && functionClient.hasFunction('recall');
        const isAssistantMessageWithFunctionRequest = (lastMessage === null || lastMessage === void 0 ? void 0 : lastMessage.message.role) === _types.MessageRole.Assistant && !!(lastMessage !== null && lastMessage !== void 0 && (_lastMessage$message$ = lastMessage.message.function_call) !== null && _lastMessage$message$ !== void 0 && _lastMessage$message$.name);
        if (recallFirst) {
          const addedMessage = {
            '@timestamp': new Date().toISOString(),
            message: {
              role: _types.MessageRole.Assistant,
              content: '',
              function_call: {
                name: 'recall',
                arguments: JSON.stringify({
                  queries: [],
                  contexts: []
                }),
                trigger: _types.MessageRole.Assistant
              }
            }
          };
          await write({
            type: _conversation_complete.StreamingChatResponseEventType.MessageAdd,
            id: (0, _uuid.v4)(),
            message: addedMessage
          });
          return await next(nextMessages.concat(addedMessage));
        } else if (isUserMessage) {
          const {
            message
          } = await (0, _handle_llm_response.handleLlmResponse)({
            signal,
            write,
            source$: (0, _stream_into_observable.streamIntoObservable)(await this.chat({
              messages: nextMessages,
              connectorId,
              stream: true,
              signal,
              functions: numFunctionsCalled >= MAX_FUNCTION_CALLS ? [] : functionClient.getFunctions().map(fn => (0, _lodash.pick)(fn.definition, 'name', 'description', 'parameters'))
            })).pipe((0, _process_openai_stream.processOpenAiStream)())
          });
          return await next(nextMessages.concat({
            message,
            '@timestamp': new Date().toISOString()
          }));
        }
        if (isAssistantMessageWithFunctionRequest) {
          const functionResponse = numFunctionsCalled >= MAX_FUNCTION_CALLS ? {
            content: {
              error: {},
              message: 'Function limit exceeded, ask the user what to do next'
            }
          } : await functionClient.executeFunction({
            connectorId,
            name: lastMessage.message.function_call.name,
            messages: nextMessages,
            args: lastMessage.message.function_call.arguments,
            signal
          }).then(response => {
            if ((0, _rxjs.isObservable)(response)) {
              return response;
            }
            const encoded = (0, _gptTokenizer.encode)(JSON.stringify(response.content || {}));
            if (encoded.length <= MAX_FUNCTION_RESPONSE_TOKEN_COUNT) {
              return response;
            }
            return {
              data: response.data,
              content: {
                message: 'Function response exceeded the maximum length allowed and was truncated',
                truncated: (0, _gptTokenizer.decode)((0, _lodash.take)(encoded, MAX_FUNCTION_RESPONSE_TOKEN_COUNT))
              }
            };
          }).catch(error => {
            return {
              content: {
                message: error.toString(),
                error
              }
            };
          });
          numFunctionsCalled++;
          if (signal.aborted) {
            return;
          }
          const functionResponseIsObservable = (0, _rxjs.isObservable)(functionResponse);
          const functionResponseMessage = {
            '@timestamp': new Date().toISOString(),
            message: {
              name: lastMessage.message.function_call.name,
              ...(functionResponseIsObservable ? {
                content: '{}'
              } : {
                content: JSON.stringify(functionResponse.content || {}),
                data: functionResponse.data ? JSON.stringify(functionResponse.data) : undefined
              }),
              role: _types.MessageRole.User
            }
          };
          nextMessages = nextMessages.concat(functionResponseMessage);
          await write({
            type: _conversation_complete.StreamingChatResponseEventType.MessageAdd,
            message: functionResponseMessage,
            id: (0, _uuid.v4)()
          });
          if (functionResponseIsObservable) {
            const {
              message
            } = await (0, _handle_llm_response.handleLlmResponse)({
              signal,
              write,
              source$: functionResponse
            });
            return await next(nextMessages.concat({
              '@timestamp': new Date().toISOString(),
              message
            }));
          }
          return await next(nextMessages);
        }
        if (!persist) {
          stream.end();
          return;
        }

        // store the updated conversation and close the stream
        if (conversationId) {
          const conversation = await this.getConversationWithMetaFields(conversationId);
          if (!conversation) {
            throw new _conversation_complete.ConversationNotFoundError();
          }
          if (signal.aborted) {
            return;
          }
          const updatedConversation = await this.update((0, _lodash.merge)({}, (0, _lodash.omit)(conversation._source, 'messages'), {
            messages: nextMessages
          }));
          await write({
            type: _conversation_complete.StreamingChatResponseEventType.ConversationUpdate,
            conversation: updatedConversation.conversation
          });
        } else {
          const generatedTitle = await titlePromise;
          if (signal.aborted) {
            return;
          }
          const conversation = await this.create({
            '@timestamp': new Date().toISOString(),
            conversation: {
              title: generatedTitle || title || 'New conversation'
            },
            messages: nextMessages,
            labels: {},
            numeric_labels: {},
            public: false
          });
          await write({
            type: _conversation_complete.StreamingChatResponseEventType.ConversationCreate,
            conversation: conversation.conversation
          });
        }
        stream.end();
      };
      next(messages).catch(error => {
        if (!signal.aborted) {
          this.dependencies.logger.error(error);
        }
        fail(error);
      });
      const titlePromise = !conversationId && !title && persist ? this.getGeneratedTitle({
        messages,
        connectorId,
        signal
      }).catch(error => {
        this.dependencies.logger.error('Could not generate title, falling back to default title');
        this.dependencies.logger.error(error);
        return Promise.resolve(undefined);
      }) : Promise.resolve(undefined);
      signal.addEventListener('abort', () => {
        stream.end();
      });
      return stream;
    });
    (0, _defineProperty2.default)(this, "chat", async ({
      messages,
      connectorId,
      functions,
      functionCall,
      stream = true,
      signal
    }) => {
      const messagesForOpenAI = (0, _lodash.compact)(messages.filter(message => {
        var _message$message$func;
        return message.message.content || ((_message$message$func = message.message.function_call) === null || _message$message$func === void 0 ? void 0 : _message$message$func.name);
      }).map(message => {
        var _message$message$func2;
        const role = message.message.role === _types.MessageRole.Elastic ? _types.MessageRole.User : message.message.role;
        return {
          role,
          content: message.message.content,
          function_call: (0, _lodash.isEmpty)((_message$message$func2 = message.message.function_call) === null || _message$message$func2 === void 0 ? void 0 : _message$message$func2.name) ? undefined : (0, _lodash.omit)(message.message.function_call, 'trigger'),
          name: message.message.name
        };
      }));
      const functionsForOpenAI = functions;
      const request = {
        messages: messagesForOpenAI,
        ...(stream ? {
          stream: true
        } : {}),
        ...(!!(functions !== null && functions !== void 0 && functions.length) ? {
          functions: functionsForOpenAI
        } : {}),
        temperature: 0,
        function_call: functionCall ? {
          name: functionCall
        } : undefined
      };
      this.dependencies.logger.debug(`Sending conversation to connector`);
      this.dependencies.logger.trace(JSON.stringify(request, null, 2));
      const executeResult = await this.dependencies.actionsClient.execute({
        actionId: connectorId,
        params: {
          subAction: stream ? 'stream' : 'run',
          subActionParams: {
            body: JSON.stringify(request),
            ...(stream ? {
              stream: true
            } : {})
          }
        }
      });
      if (executeResult.status === 'error') {
        throw (0, _boom.internal)(`${executeResult === null || executeResult === void 0 ? void 0 : executeResult.message} - ${executeResult === null || executeResult === void 0 ? void 0 : executeResult.serviceMessage}`);
      }
      const response = stream ? executeResult.data : executeResult.data;
      if (response instanceof _stream.Readable) {
        signal.addEventListener('abort', () => response.destroy());
      }
      return response;
    });
    (0, _defineProperty2.default)(this, "find", async options => {
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: this.dependencies.resources.aliases.conversations,
        allow_no_indices: true,
        query: {
          bool: {
            filter: [...(0, _get_access_query.getAccessQuery)({
              user: this.dependencies.user,
              namespace: this.dependencies.namespace
            })]
          }
        },
        sort: {
          '@timestamp': 'desc'
        },
        size: 100
      });
      return {
        conversations: response.hits.hits.map(hit => hit._source)
      };
    });
    (0, _defineProperty2.default)(this, "update", async conversation => {
      const document = await this.getConversationWithMetaFields(conversation.conversation.id);
      if (!document) {
        throw (0, _boom.notFound)();
      }
      const updatedConversation = (0, _lodash.merge)({}, conversation, this.getConversationUpdateValues(new Date().toISOString()));
      await this.dependencies.esClient.asInternalUser.update({
        id: document._id,
        index: document._index,
        doc: updatedConversation,
        refresh: true
      });
      return updatedConversation;
    });
    (0, _defineProperty2.default)(this, "getGeneratedTitle", async ({
      messages,
      connectorId,
      signal
    }) => {
      var _response$message;
      const stream = await this.chat({
        messages: [{
          '@timestamp': new Date().toISOString(),
          message: {
            role: _types.MessageRole.User,
            content: messages.slice(1).reduce((acc, curr) => {
              return `${acc} ${curr.message.role}: ${curr.message.content}`;
            }, 'You are a helpful assistant for Elastic Observability. Assume the following message is the start of a conversation between you and a user; give this conversation a title based on the content below. DO NOT UNDER ANY CIRCUMSTANCES wrap this title in single or double quotes. This title is shown in a list of conversations to the user, so title it for the user, not for you. Here is the content:')
          }
        }],
        connectorId,
        stream: true,
        signal
      });
      const response = await (0, _rxjs.lastValueFrom)((0, _stream_into_observable.streamIntoObservable)(stream).pipe((0, _process_openai_stream.processOpenAiStream)(), (0, _concatenate_openai_chunks.concatenateOpenAiChunks)()));
      const input = ((_response$message = response.message) === null || _response$message === void 0 ? void 0 : _response$message.content) || '';

      // This regular expression captures a string enclosed in single or double quotes.
      // It extracts the string content without the quotes.
      // Example matches:
      // - "Hello, World!" => Captures: Hello, World!
      // - 'Another Example' => Captures: Another Example
      // - JustTextWithoutQuotes => Captures: JustTextWithoutQuotes
      const match = input.match(/^["']?([^"']+)["']?$/);
      const title = match ? match[1] : input;
      return title;
    });
    (0, _defineProperty2.default)(this, "setTitle", async ({
      conversationId,
      title
    }) => {
      const document = await this.getConversationWithMetaFields(conversationId);
      if (!document) {
        throw (0, _boom.notFound)();
      }
      const conversation = await this.get(conversationId);
      if (!conversation) {
        throw (0, _boom.notFound)();
      }
      const updatedConversation = (0, _lodash.merge)({}, conversation, {
        conversation: {
          title
        }
      }, this.getConversationUpdateValues(new Date().toISOString()));
      await this.dependencies.esClient.asInternalUser.update({
        id: document._id,
        index: document._index,
        doc: {
          conversation: {
            title
          }
        },
        refresh: true
      });
      return updatedConversation;
    });
    (0, _defineProperty2.default)(this, "create", async conversation => {
      const now = new Date().toISOString();
      const createdConversation = (0, _lodash.merge)({}, conversation, {
        '@timestamp': now,
        conversation: {
          id: (0, _uuid.v4)()
        }
      }, this.getConversationUpdateValues(now));
      await this.dependencies.esClient.asInternalUser.index({
        index: this.dependencies.resources.aliases.conversations,
        document: createdConversation,
        refresh: true
      });
      return createdConversation;
    });
    (0, _defineProperty2.default)(this, "recall", async ({
      queries,
      contexts
    }) => {
      return this.dependencies.knowledgeBaseService.recall({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        queries,
        contexts,
        asCurrentUser: this.dependencies.esClient.asCurrentUser
      });
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseStatus", () => {
      return this.dependencies.knowledgeBaseService.status();
    });
    (0, _defineProperty2.default)(this, "setupKnowledgeBase", () => {
      return this.dependencies.knowledgeBaseService.setup();
    });
    (0, _defineProperty2.default)(this, "createKnowledgeBaseEntry", async ({
      entry
    }) => {
      return this.dependencies.knowledgeBaseService.addEntry({
        namespace: this.dependencies.namespace,
        user: this.dependencies.user,
        entry
      });
    });
    (0, _defineProperty2.default)(this, "importKnowledgeBaseEntries", async ({
      entries
    }) => {
      const operations = entries.map(entry => ({
        type: _knowledge_base_service.KnowledgeBaseEntryOperationType.Index,
        document: {
          ...entry,
          '@timestamp': new Date().toISOString()
        }
      }));
      await this.dependencies.knowledgeBaseService.addEntries({
        operations
      });
    });
    (0, _defineProperty2.default)(this, "getKnowledgeBaseEntries", async ({
      query,
      sortBy,
      sortDirection
    }) => {
      return this.dependencies.knowledgeBaseService.getEntries({
        query,
        sortBy,
        sortDirection
      });
    });
    (0, _defineProperty2.default)(this, "deleteKnowledgeBaseEntry", async id => {
      return this.dependencies.knowledgeBaseService.deleteEntry({
        id
      });
    });
    this.dependencies = dependencies;
  }
}
exports.ObservabilityAIAssistantClient = ObservabilityAIAssistantClient;