"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetDatasetInfoFunction = registerGetDatasetInfoFunction;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _types = require("../../common/types");
var _concatenate_openai_chunks = require("../../common/utils/concatenate_openai_chunks");
var _process_openai_stream = require("../../common/utils/process_openai_stream");
var _stream_into_observable = require("../service/util/stream_into_observable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerGetDatasetInfoFunction({
  client,
  resources,
  registerFunction
}) {
  registerFunction({
    name: 'get_dataset_info',
    contexts: ['core'],
    visibility: _types.FunctionVisibility.System,
    description: `Use this function to get information about indices/datasets available and the fields available on them.

      providing empty string as index name will retrieve all indices
      else list of all fields for the given index will be given. if no fields are returned this means no indices were matched by provided index pattern.
      wildcards can be part of index name.`,
    descriptionForUser: 'This function allows the assistant to get information about available indices and their fields.',
    parameters: {
      type: 'object',
      additionalProperties: false,
      properties: {
        index: {
          type: 'string',
          description: 'index pattern the user is interested in or empty string to get information about all available indices'
        }
      },
      required: ['index']
    }
  }, async ({
    arguments: {
      index
    },
    messages,
    connectorId
  }, signal) => {
    const coreContext = await resources.context.core;
    const esClient = coreContext.elasticsearch.client.asCurrentUser;
    const savedObjectsClient = coreContext.savedObjects.getClient();
    let indices = [];
    try {
      const body = await esClient.indices.resolveIndex({
        name: index === '' ? '*' : index,
        expand_wildcards: 'open'
      });
      indices = [...body.indices.map(i => i.name), ...body.data_streams.map(d => d.name)];
    } catch (e) {
      indices = [];
    }
    if (index === '') {
      return {
        content: {
          indices,
          fields: []
        }
      };
    }
    if (indices.length === 0) {
      return {
        content: {
          indices,
          fields: []
        }
      };
    }
    const fields = await resources.plugins.dataViews.start().then(dataViewsStart => dataViewsStart.dataViewsServiceFactory(savedObjectsClient, esClient)).then(service => service.getFieldsForWildcard({
      pattern: index
    }));

    // else get all the fields for the found dataview
    const response = {
      indices: [index],
      fields: fields.flatMap(field => {
        var _field$esTypes;
        return ((_field$esTypes = field.esTypes) !== null && _field$esTypes !== void 0 ? _field$esTypes : [field.type]).map(type => {
          return {
            name: field.name,
            description: field.customLabel || '',
            type
          };
        });
      })
    };
    const allFields = response.fields;
    const fieldNames = (0, _lodash.uniq)(allFields.map(field => field.name));
    const groupedFields = (0, _lodash.groupBy)(allFields, field => field.name);
    const relevantFields = await Promise.all((0, _lodash.chunk)(fieldNames, 500).map(async fieldsInChunk => {
      var _chunkResponse$messag, _chunkResponse$messag2, _chunkResponse$messag3, _chunkResponse$messag4;
      const chunkResponse$ = (0, _stream_into_observable.streamIntoObservable)(await client.chat({
        connectorId,
        signal,
        messages: [{
          '@timestamp': new Date().toISOString(),
          message: {
            role: _types.MessageRole.System,
            content: `You are a helpful assistant for Elastic Observability.
                  Your task is to create a list of field names that are relevant
                  to the conversation, using ONLY the list of fields and
                  types provided in the last user message. DO NOT UNDER ANY
                  CIRCUMSTANCES include fields not mentioned in this list.`
          }
        }, ...messages.slice(1), {
          '@timestamp': new Date().toISOString(),
          message: {
            role: _types.MessageRole.User,
            content: `This is the list:

                  ${fieldsInChunk.join('\n')}`
          }
        }],
        functions: [{
          name: 'fields',
          description: 'The fields you consider relevant to the conversation',
          parameters: {
            type: 'object',
            additionalProperties: false,
            properties: {
              fields: {
                type: 'array',
                additionalProperties: false,
                items: {
                  type: 'string',
                  additionalProperties: false
                }
              }
            },
            required: ['fields']
          }
        }],
        functionCall: 'fields',
        stream: true
      })).pipe((0, _process_openai_stream.processOpenAiStream)(), (0, _concatenate_openai_chunks.concatenateOpenAiChunks)());
      const chunkResponse = await (0, _rxjs.lastValueFrom)(chunkResponse$);
      return (_chunkResponse$messag = chunkResponse.message) !== null && _chunkResponse$messag !== void 0 && (_chunkResponse$messag2 = _chunkResponse$messag.function_call) !== null && _chunkResponse$messag2 !== void 0 && _chunkResponse$messag2.arguments ? JSON.parse(chunkResponse.message.function_call.arguments).fields.filter(field => fieldsInChunk.includes(field)).map(field => {
        const fieldDescriptors = groupedFields[field];
        return `${field}:${fieldDescriptors.map(descriptor => descriptor.type).join(',')}`;
      }) : [(_chunkResponse$messag3 = (_chunkResponse$messag4 = chunkResponse.message) === null || _chunkResponse$messag4 === void 0 ? void 0 : _chunkResponse$messag4.content) !== null && _chunkResponse$messag3 !== void 0 ? _chunkResponse$messag3 : ''];
    }));
    return {
      content: {
        indices: response.indices,
        fields: relevantFields.flat()
      }
    };
  });
}